'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on 2011-06-09

@author: Jimmy Li
'''

from Tkinter import *
from control.DataBoard import DataBoard
import gui.Utils
import gui.dialogs.EditConstraint
import tkMessageBox
import sys

class EditorFrame(Toplevel):
    '''
    classdocs
    '''

    def __init__(self, master, geoX = None, geoY = None):
        '''
        Constructor
        '''
        Toplevel.__init__(self, master)
        self.master = master
        self.title("Waypoint Text Editor")
        self.listBoxWidth = 12
        if sys.platform == 'win32':
            self.listBoxWidth = 15
        self.listBoxHeight = 13 
        
        self.wptCmdWidth = 40 
        self.wptCmdHeight = 24
        #self.wptListWidth = 21
        self.wptListHeight = 13
        self.standardOptionWidth = 15
        self.launchExtEditorEntryWidth = 20
        self.wptPosWidth = 23
        self.wptPosHeight = 13
        
        self.editingWayptSet = None
        self.editingWaypt = {}
        self.positionNames = []
        
        self.designerSelectionHighlight = "#FF9966"
        self.selectionHighlight = "#A8D4FF"
        self.selectionTextColor = "black"
        
        self.protocol("WM_DELETE_WINDOW", self.closeWindow)
        
        self.resizable(width=TRUE, height=TRUE)
        
        if geoX == None:
            geoX = master.winfo_rootx()+720
        
        if geoY == None:
            geoY = master.winfo_rooty()+20
            
        self.geometry("+%d+%d" % (geoX, geoY))
        
        #self.tk.call('wm', 'iconphoto', self._w, DataBoard.icon)
        
        self.bind("<Control-KeyRelease-z>", gui.Utils.undo)
        self.bind("<Control-KeyRelease-r>", gui.Utils.redo)
        
        # MASTER FRAME
        wayptMasterFrame = Frame(self, background="#F0F0F0")
        wayptMasterFrame.grid(row=0, column=0, sticky=(N,S,E,W), ipady=3, ipadx=5)
        self.columnconfigure(0, weight=1)
        self.rowconfigure(0, weight=1)
        
        
        # WAYPOINT SET LIST FRAME
        wayptListFrame = Frame(wayptMasterFrame)
        wayptListFrame.grid(row=0, column=0, sticky=(N,S), padx=10, pady=5)
        wayptMasterFrame.rowconfigure(0, weight=1)
        wayptMasterFrame.rowconfigure(1, weight=1)
        
        scrollbar = Scrollbar(wayptListFrame, orient=VERTICAL)
        self.wayptListBox = Listbox(wayptListFrame, selectbackground=self.selectionHighlight, exportselection=0, height=self.listBoxHeight, width=self.listBoxWidth, selectmode=SINGLE, yscrollcommand=scrollbar.set)
        scrollbar.config(command=self.wayptListBox.yview)
        scrollbar.pack(side=RIGHT, fill=Y)
        self.wayptListBox.pack(side=TOP, fill=BOTH, expand=1)
        self.wayptListBox.insert(END, "- WPT SETS -------")        
        self.wayptListBox.bind("<<ListboxSelect>>", self.clickWayptSetList)
        
        
        # WAYPOINTS LIST FRAME
        wayptsListFrame = Frame(wayptMasterFrame)
        wayptsListFrame.grid(row=1, column=0, padx=10, pady=5, sticky=(N, S))
        scrollbar = Scrollbar(wayptsListFrame, orient=VERTICAL) 
        self.wayptsListBox = Listbox(wayptsListFrame, selectbackground=self.selectionHighlight, exportselection=0, width=self.listBoxWidth, height=self.wptListHeight, selectmode=SINGLE, yscrollcommand=scrollbar.set)
        scrollbar.config(command=self.wayptsListBox.yview)
        scrollbar.pack(side=RIGHT, fill=Y)
        self.wayptsListBox.pack(side=LEFT, fill=BOTH, expand=1)        
        self.wayptsListBox.insert(END, "- WAYPOINTS ------")        
        self.wayptsListBox.bind("<<ListboxSelect>>", self.clickWaypt)
        
        
        # COMMAND EDITOR FRAME
        wayptCmdFrame = Frame(wayptMasterFrame, background="#F0F0F0")
        wayptCmdFrame.grid(row=0, column=1, padx=5, pady=5, sticky=(N,S,E,W), rowspan=2)
        wayptMasterFrame.columnconfigure(1, weight=1)
        wayptCmdFrame.rowconfigure(1, weight=1)
        wayptCmdFrame.columnconfigure(0, weight=1)
        
        # --- COMMANDS
        Label(wayptCmdFrame, text="Code", background="#F0F0F0").grid(row=0, column=0, columnspan=2, sticky=W, padx=0)
        scrollbar = Scrollbar(wayptCmdFrame)
        scrollbar.grid(row=1, column=1, sticky=(N,S), pady=5)
        self.wayptCommandEditor = Text(wayptCmdFrame, width=self.wptCmdWidth, height=self.wptCmdHeight, yscrollcommand=scrollbar.set)
        self.wayptCommandEditor.grid(row=1, column=0, sticky=(N,S,E,W), padx=0, pady=5)        
        self.wayptCommandEditor.config(state=DISABLED)
        scrollbar.config(command=self.wayptCommandEditor.yview)
        self.wayptCommandEditor.bind("<KeyRelease>", self.updateWayptCommand)
        
        
        # LAUNCH EXTERNAL EDITOR FRAME
        launchExtEditorFrame = Frame(wayptCmdFrame, background="#F0F0F0")
        launchExtEditorFrame.grid(row=2, column=0, padx=5, pady=5, sticky=(E,W))
        launchExtEditorFrame.columnconfigure(0,weight=1)
        
        # --- LAUNCH EXTERNAL EDITOR
        self.extEditorEntry = Entry(launchExtEditorFrame, width = self.launchExtEditorEntryWidth)
        self.extEditorEntry.bind("<Return>", self.launchExtEditor)
        self.extEditorEntry.grid(row=0, column=0, sticky=(E,W))
        if DataBoard.rcData.has_key('text_editor'):
            self.extEditorEntry.insert(0,DataBoard.rcData['text_editor'])
        self.launchExtEditorButton = Button(launchExtEditorFrame, text="Launch Editor", command=self.launchExtEditor)
        self.launchExtEditorButton.grid(row=0,column=1, padx=5)
        
        
        # OPTIONS EDITOR FRAME
        wayptEditorFrame = Frame(wayptMasterFrame, background="#F0F0F0")
        wayptEditorFrame.grid(row=0, column=2, padx=5, pady=5, sticky=(N,S,E,W), rowspan=2)
        wayptEditorFrame.rowconfigure(1,weight=1)
        
        # --- NAME
        Label(wayptEditorFrame, text="Set Name", background="#F0F0F0").grid(row=0, column=0, sticky=W, padx=5, pady=3)
        self.wayptNameEditor = Entry(wayptEditorFrame, width=self.standardOptionWidth)
        self.wayptNameEditor.grid(row=0, column=1, sticky=W, padx=5)
        self.wayptNameEditor.config(state=DISABLED)
        self.wayptNameEditor.bind("<KeyRelease>", self.updateWayptName)
        
        # --- POSITIONS
        wayptPositionDisplayFrame = Frame(wayptEditorFrame)
        wayptPositionDisplayFrame.grid(row=1, columnspan=2, column=0, sticky=(N,S), padx=4, pady=5)
        scrollbar2 = Scrollbar(wayptPositionDisplayFrame, orient=VERTICAL)
        self.wayptPositionDisplay = Listbox(wayptPositionDisplayFrame, height=self.wptPosHeight, width=self.wptPosWidth, yscrollcommand=scrollbar2.set, exportselection=0)
        scrollbar2.config(command=self.wayptPositionDisplay.yview)
        scrollbar2.pack(side=RIGHT, fill=Y)
        self.wayptPositionDisplay.pack(side=TOP, fill=BOTH, expand=1)
        self.wayptPositionDisplay.insert(END, "- POSITION ------------------------------")
        self.wayptPositionDisplay.bind("<<ListboxSelect>>", self.clickPosition)
        
        # ---- DELETING
        buttonFrame = Frame(wayptEditorFrame)
        buttonFrame.grid(row=3, column=0, pady=5, padx=5, sticky=W, columnspan=2)
        self.deleteWayptSetButton = Button(buttonFrame, text="Delete Set", command=self.deleteWayptSet)
        self.deleteWayptSetButton.pack(side=LEFT)
        self.deleteWayptButton = Button(buttonFrame, text="Delete Waypt", command=self.deleteWaypt)
        self.deleteWayptButton.pack(side=LEFT)
        
        self.refreshWayptSetList()
        self.refreshWayptsList()
        self.updateSelectedWaypt()       
    
    def updateWayptName(self, event):
        if self.editingWayptSet == None:
            return
        val = self.wayptNameEditor.get()
        if self.editingWayptSet.name == val:
            return
        self.editingWayptSet.name = val 
        gui.Utils.modified()
        self.refreshWayptSetList()
        gui.Utils.updateDesignerStatusBars()
        '''
        for designer in DataBoard.wptEditors:
            designer.refreshWayptSetList()
            if not designer is self and designer.editingWayptSet is self.editingWayptSet:
                designer.refreshWayptName()
        '''
    
    def clickWayptSetList(self,event):
        sel = self.wayptListBox.curselection()
        if len(sel) == 0:
            return
        index = sel[0]
        if index == '0':
            return
        wayptset = self.wayptListBoxData[int(index)-1]
        self.editingWayptSet = wayptset
        
        if self.editingWaypt.has_key(self.editingWayptSet.id):
            remembered = self.editingWaypt[self.editingWayptSet.id]
            if remembered >= len(self.editingWayptSet.waypoints):
                self.editingWaypt[self.editingWayptSet.id] = 0
        else:
            self.editingWaypt[self.editingWayptSet.id] = 0
            
        DataBoard.selectedWaypointSet = wayptset.id
        DataBoard.selectedWaypoint = self.editingWaypt[self.editingWayptSet.id] + 1
        
        gui.Utils.redrawAllDesigners()
        
        gui.Utils.updateWaypointEditors()
        self.updateSelectedWaypt()
    
    def clickWaypt(self, event):
        sel = self.wayptsListBox.curselection()
        if len(sel) == 0:
            return
        index = sel[0]
        if index == '0':
            return
        self.editingWaypt[self.editingWayptSet.id] = int(sel[0])-1
        DataBoard.selectedWaypoint = int(sel[0])
        self.updateSelectedWaypt()
        gui.Utils.redrawAllDesigners()
        '''
        for designer in DataBoard.wptEditors:
            if self.editingWayptSet is designer.editingWayptSet:
                designer.refreshWayptsList()
        '''
    
    def launchExtEditor(self, event=None):
        if self.editingWayptSet == None:
            #print 'area1'
            return
        waypt = self.editingWayptSet.waypoints[self.editingWaypt[self.editingWayptSet.id]]
        if DataBoard.extEditorManager.files.has_key(waypt.id):
            DataBoard.extEditorManager.removeFile(waypt)
            #del waypt.temp['extEdit']
            self.refreshWayptCommand()
            gui.Utils.modified(atomic=True)
            #print 'area2'
        else:
            #print 'area3'
            result = DataBoard.extEditorManager.addFile(waypt, self.extEditorEntry.get(), self.editingWayptSet.name+"_Waypoint_"+str(self.editingWaypt[self.editingWayptSet.id]))
            if result == '1':
                #waypt.temp['extEdit'] = True
                self.refreshWayptCommand()
            else:
                tkMessageBox.showerror("Error", result)
                
    def clickPosition(self, event):
        #d = gui.dialogs.DesignerSettings.DesignerSettings1D(self)
        sel = self.wayptPositionDisplay.curselection()
        if len(sel) == 0:
            return
        index = sel[0]
        if index == '0':
            return
        
        positionDic = self.editingWayptSet.waypoints[self.editingWaypt[self.editingWayptSet.id]].position
        name = self.positionNames[int(sel[0])-1]
        
        d = gui.dialogs.EditConstraint.Position(self, name, positionDic[name])
        
        if d.result == None:
            return
        
        positionDic[name] = d.result
        
        gui.Utils.modified(atomic=True)
        
        self.refreshWayptPosition()
        gui.Utils.redrawAllDesigners()
    
    def deleteWayptSet(self):
        del DataBoard.waypointSets[DataBoard.selectedWaypointSet]
        DataBoard.selectedWaypointSet = ''
        DataBoard.selectedWaypoint = -1
        del self.editingWaypt[self.editingWayptSet.id]
        self.editingWayptSet = None
        
        gui.Utils.modified(atomic=True)
        
        gui.Utils.updateWaypointEditors()
        gui.Utils.reinitAllDesigners()
    
    def deleteWaypt(self):
        if len(DataBoard.waypointSets[DataBoard.selectedWaypointSet].waypoints) == 1:
            self.deleteWayptSet()
            return
        
        del DataBoard.waypointSets[DataBoard.selectedWaypointSet].waypoints[int(DataBoard.selectedWaypoint)-1]
        DataBoard.selectedWaypoint = int(DataBoard.selectedWaypoint) - 1
        if DataBoard.selectedWaypoint < 1:
            DataBoard.selectedWaypoint = 1
        self.editingWaypt[self.editingWayptSet.id] = DataBoard.selectedWaypoint - 1
        
        gui.Utils.modified(atomic=True)
        
        gui.Utils.redrawAllDesigners()
        gui.Utils.updateWaypointEditors()
        
    def updateWayptCommand(self, event):
        if self.editingWayptSet == None:
            return
        val = self.wayptCommandEditor.get(1.0, END)
        if self.editingWayptSet.waypoints[self.editingWaypt[self.editingWayptSet.id]].commands.strip() == val.strip():
            return
        self.editingWayptSet.waypoints[self.editingWaypt[self.editingWayptSet.id]].commands = val 
        
        gui.Utils.modified()
        '''
        for designer in DataBoard.wptEditors:
            if not designer is self and designer.editingWayptSet is self.editingWayptSet:
                designer.refreshWayptCommand()
        '''
        
    def refreshWayptSetList(self):
        self.wayptListBox.delete(1, END)
        self.wayptListBoxData = []
        self.editingWayptSet = None
        
        for wpsID in DataBoard.waypointSets:
            wps = DataBoard.waypointSets[wpsID]
            self.wayptListBoxData.append(wps)
            
        self.wayptListBoxData = sorted(self.wayptListBoxData, key=lambda wayptset: wayptset.name)
        
        selectedIndex = 0
        for wps in self.wayptListBoxData:
            self.wayptListBox.insert(END, wps.name)
            if wps.id == DataBoard.selectedWaypointSet:
                self.editingWayptSet = wps
                selectedIndex = self.wayptListBox.size() - 1
        
        if selectedIndex > 0:
            self.wayptListBox.selection_set(first=selectedIndex)
            
        
    def refreshWayptsList(self):
        if self.editingWayptSet == None:
            return
        self.wayptsListBox.delete(1, END)
        for wpt in range(0, len(self.editingWayptSet.waypoints)):
            self.wayptsListBox.insert(END, "Waypoint %d" % (wpt+1))
        
        self.editingWaypt[self.editingWayptSet.id] = int(DataBoard.selectedWaypoint) - 1
        self.wayptsListBox.selection_set(first=self.editingWaypt[self.editingWayptSet.id]+1)
        
    def updateSelectedWaypt(self):
        if DataBoard.selectedWaypointSet == '' or DataBoard.selectedWaypoint < 1:
            self.wayptNameEditor.delete(0, END)
            self.wayptNameEditor.configure(state=DISABLED)
            self.wayptsListBox.delete(1, END)
            self.wayptCommandEditor.delete(1.0, END)
            self.wayptCommandEditor.configure(state=DISABLED)
            self.wayptPositionDisplay.delete(1, END)
            self.extEditorEntry.configure(state=DISABLED)
            self.launchExtEditorButton.configure(state=DISABLED)
            self.deleteWayptButton.configure(state=DISABLED)
            self.deleteWayptSetButton.configure(state=DISABLED)
            return
        
        self.refreshWayptsList()
        self.wayptNameEditor.config(state=NORMAL)
        #self.wayptCommandEditor.configure(state=NORMAL)
        #self.extEditorEntry.configure(state=NORMAL)
        #self.launchExtEditorButton.configure(state=NORMAL)
        self.deleteWayptButton.configure(state=NORMAL)
        self.deleteWayptSetButton.configure(state=NORMAL)
        self.refreshWayptName()
        self.refreshWayptCommand()
        self.refreshWayptPosition()
        
    def refreshWayptName(self):
        self.wayptNameEditor.delete(0, END)
        self.wayptNameEditor.insert(0, self.editingWayptSet.name)
    
    def refreshWayptCommand(self):
        waypt = self.editingWayptSet.waypoints[self.editingWaypt[self.editingWayptSet.id]]
        if DataBoard.extEditorManager.files.has_key(waypt.id):
            #print 'area5'
            self.wayptCommandEditor.config(state=NORMAL)
            self.wayptCommandEditor.delete(1.0, END)
            self.wayptCommandEditor.insert(END, self.editingWayptSet.waypoints[self.editingWaypt[self.editingWayptSet.id]].commands)
            self.wayptCommandEditor.config(state=DISABLED)
            self.extEditorEntry.configure(state=DISABLED)
            self.launchExtEditorButton.configure(state=NORMAL, text="Release Editor")
        else:
            #print 'area6'
            self.wayptCommandEditor.config(state=NORMAL)
            self.wayptCommandEditor.delete(1.0, END)
            self.wayptCommandEditor.insert(END, self.editingWayptSet.waypoints[self.editingWaypt[self.editingWayptSet.id]].commands)
            self.extEditorEntry.configure(state=NORMAL)
            self.launchExtEditorButton.configure(state=NORMAL, text="Attach Editor")
        
        
    def refreshWayptPosition(self):
        self.wayptPositionDisplay.delete(1, END)
        positionNames = []
        positionDic = self.editingWayptSet.waypoints[self.editingWaypt[self.editingWayptSet.id]].position
        for pos in positionDic:
            positionNames.append(pos)
        positionNames = sorted(positionNames)
        for name in positionNames:
            self.wayptPositionDisplay.insert(END, name+": "+"%.2f" % positionDic[name])
        self.positionNames = positionNames
    
    def closeWindow(self):
        gui.Utils.removeWptEditor(self)
        self.destroy()
