'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''


'''
Created on 2010-05-25

@author: Jimmy Li
'''

import os
import sys
import time
import urllib
import pickle
from Tkinter import *
from control.DataBoard import DataBoard
import control.Utils
import gui.DesignerTK.OneD
import gui.DesignerTK.TwoD
import gui.TextEditor.RegionEditor
import gui.TextEditor.WaypointEditor
import gui.TextEditor.BooleanEditor
import gui.LogPlayback
import gui.Utils
from gui.dialogs import DesignerSettings, MetaController, NewDesign, AttachMetaController
from gui.controls.LogPlaybackControl import PlaybackController
import tkFileDialog
import tkMessageBox
import StatusBar

class MainWindow():
    '''
    Main window displaying menu
    '''
    #============= CONSTANTS ==============================
    #Window
    width = 650
    height = 32
    x = 40
    y = 40

    if sys.platform == 'win32':
        height = 55

    updateCheckInterval = 2628000 # A month 2628000; day: 86400
    
    def __init__(self):
        '''
        Constructor
        '''
        self.designers = []
        self.editingRegion = None
        self.regionListBoxData = []
        self.editingRegionIndex = 0
        self.editingWayptSet = None
        self.wayptListBoxData = []
        self.editingWayptSetIndex = 0
        self.title = "Graphical State Space Programming"
        self.fileOptions = {}
        self.fileOptions['defaultextension'] = '.gssp'
        self.fileOptions['filetypes'] = [('gssp save files', '.gssp'), ('all files', '.*')]
        self.logOptions = {}
        self.logOptions['defaultextension'] = '.gssplog'
        self.logOptions['filetypes'] = [('gssp log files', '.gssplog'), ('all files', '.*')]
        self.moduleOptions = {}
        self.moduleOptions['defaultextension'] = '.gsspmod'
        self.moduleOptions['filetypes'] = [('gssp module', '.gsspmod'), ('all files', '.*')]
        
    def createWindow(self):
        root = Tk()
        root.geometry('%dx%d+%d+%d' % (MainWindow.width, MainWindow.height, MainWindow.x, MainWindow.y))
        root.resizable(width=TRUE, height=TRUE)
        root.configure(background="#F0F0F0")
        root.title(self.title)
        root.bind("<Control-KeyRelease-z>", gui.Utils.undo)
        root.bind("<Control-KeyRelease-r>", gui.Utils.redo)
        root.protocol("WM_DELETE_WINDOW", self.closeWindow)
        self.root = root
        
        masterFrame = Frame(root, background="#F0F0F0")
        masterFrame.grid(row=0, column=0, sticky=(N,S,E,W), ipady=3, ipadx=5)
        root.columnconfigure(0, weight=1)
        root.rowconfigure(0, weight=1)
        
        #Label(masterFrame, text="Welcome to Graphical State Space Programming", background="#F0F0F0").grid(row=0, column=0, sticky=W, padx=3, pady=3)
        self.menuButtonNew = Button(masterFrame, text="New", command=self.newRobotDesign)
        self.menuButtonNew.grid(row=0, column=0, sticky=W, padx=3, pady=3)
        self.menuButtonOpen = Button(masterFrame, text="Open", command=self.loadRobotDesign)
        self.menuButtonOpen.grid(row=0, column=1, sticky=W, padx=3, pady=3)
        self.menuButtonSave = Button(masterFrame, text="Save", command=self.saveRobotDesign, state=DISABLED)
        self.menuButtonSave.grid(row=0, column=2, sticky=W, padx=3, pady=3)
        self.menuButtonSaveAs = Button(masterFrame, text="Save As", command=self.saveAsRobotDesign, state=DISABLED)
        self.menuButtonSaveAs.grid(row=0, column=3, sticky=W, padx=3, pady=3)
        
        root.wait_visibility()
        
        self.newRegTextEditor(MainWindow.x, MainWindow.y + 100 + 20)
        self.newWptTextEditor(MainWindow.x + 720, MainWindow.y + 100 + 20)
    
    def closeWindow(self):
        if DataBoard.modified:
            r = tkMessageBox.askquestion("Exit", "You have unsaved work. You will lose them if you exit now. Are you sure you want to exit?")
            if r == 'no':
                return
        self.root.destroy()
        if not DataBoard.asyncClient == None:
            DataBoard.asyncClient.close()
    
    def createMenu(self):
        menu = Menu(self.root)
        self.root.config(menu=menu)
        self.menu = menu
        
        file = Menu(menu)
        self.fileMenu = file
        menu.add_cascade(label='File', menu=file)
        
        file.add_command(label='New', command=self.newRobotDesign)
        file.add_command(label='Open', command=self.loadRobotDesign)
        file.add_command(label='Save', command=self.saveRobotDesign, state=DISABLED)
        file.add_command(label='Save as', command=self.saveAsRobotDesign, state=DISABLED)
        
        edit = Menu(menu)
        self.editMenu = edit
        menu.add_cascade(label='Edit', menu=edit)
        edit.add_command(label='Undo (Ctrl-z)', command=gui.Utils.undo, state=DISABLED)
        edit.add_command(label='Redo (Ctrl-r)', command=gui.Utils.redo, state=DISABLED)
        
        design = Menu(menu)
        self.designMenu = design
        menu.add_cascade(label='Design', menu=design)
        design.add_command(label='Region Text Editor', command=self.newRegTextEditor, state=NORMAL)
        design.add_command(label='Waypoint Text Editor', command=self.newWptTextEditor, state=NORMAL)
        design.add_command(label='New 1D Graphical Designer', command=self.newDesigner1D, state=DISABLED)
        design.add_command(label='New 2D Graphical Designer', command=self.newDesigner2D, state=DISABLED)
        design.add_command(label='New Boolean Region Editor', command=self.newBooleanRegionEditor, state=DISABLED)
        
        run = Menu(menu)
        self.runMenu = run
        menu.add_cascade(label='Run', menu=run)
        run.add_command(label='Start Meta-controller', command=self.startMetaController, state=DISABLED)
        run.add_command(label='Attach to Meta-controller', command=self.attachMetaController, state=DISABLED)
        run.add_command(label='Detach from Meta-controller', command=self.detachMetaController, state=DISABLED)
        run.add_command(label='Stop Meta-controller', command=self.stopMetaController, state=DISABLED)
        run.add_command(label='Log File Playback', command=self.startLogPlayback, state=DISABLED)
        run.add_command(label='Close Log File Playback', command=self.stopLogPlayback, state=DISABLED)
        
        help = Menu(menu)
        self.aboutMenu = help
        menu.add_cascade(label='Help', menu=help)
        help.add_command(label='About', command=self.showAbout)
        
        if DataBoard.rcData.has_key('last_update_check'):
            last_upd_chk = float(DataBoard.rcData['last_update_check'])
            if (time.time() - last_upd_chk > MainWindow.updateCheckInterval): 
                self.checkForNewVersion()
        else:
            self.checkForNewVersion()
    
    def checkForNewVersion(self):
        print 'Performing update check.'
        
        DataBoard.rcData['last_update_check'] = time.time()
        gui.Utils.rcCleanup()
            
        try:
            page = urllib.urlopen("http://cim.mcgill.ca/gssp/current_version.txt")
        except:
            return
        data = page.read().strip()
        page.close()
        
        if len(data) > 8:
            '''
            Data doesn't look right
            '''
            return
        
        if not data.strip() == DataBoard.version: 
            tkMessageBox.showinfo("New Version Available", "A new version of GSSP is available. \n\nYour version: " + DataBoard.version + "\nNew Version: " + data + "\n\nYou can download it at:\n\nhttp://cim.mcgill.ca/gssp/")
    
    def showAbout(self):
        tkMessageBox.showinfo("About", "Graphical State Space Programming (GSSP)\n\nVersion "+DataBoard.version+"\n\nhttp://cim.mcgill.ca/gssp/")
        
    def dumpWaypoints(self):
        control.Waypoint.dumpWaypoints()        
    
    def createStatusBar(self):
        status = StatusBar.StatusBar(self.root)
        status.pack(side=BOTTOM, fill=X)
        self.statusBar = status 
        
    def newRobotDesign(self):
        if DataBoard.modified:
            r = tkMessageBox.askquestion("New", "You have unsaved work. You will lose them if you start a new design. Are you sure you want to continue?")
            if r == 'no':
                return
        
        self.detachMetaController()
        
        d =  NewDesign.NewDesignDialog(self.root)
        if d.result == None:
            return
            
        DataBoard.currentStateVarSet = d.result
        
        for des in DataBoard.designers:
            des.destroy()
        
        control.Utils.resetDataBoard()
        
        self.root.title("Unsaved - "+self.title)
        
        self.resetMenu()
        
        globalRegion = control.Region.Region("global")
        DataBoard.regions.addRegion(globalRegion)

        gui.Utils.modified()
        
        DataBoard.modified = False
        
        gui.Utils.reinitAllDesigners()
        gui.Utils.updateRegionEditors()
        gui.Utils.updateWaypointEditors() 
    
    def saveRobotDesign(self):
        if DataBoard.currentFile == '':
            self.saveAsRobotDesign()   
        else:
            control.Utils.saveDesignToFile()
        
    
    def saveAsRobotDesign(self):
        file = tkFileDialog.asksaveasfile(mode="wb", **self.fileOptions)
        if not file == None:
            control.Utils.saveDesignToFile(file)
            self.root.title(os.path.split(file.name)[1]+' - '+self.title)
            DataBoard.currentFile = file.name
            file.close()
            
    def loadRobotDesign(self):
        if DataBoard.modified:
            r = tkMessageBox.askquestion("Open", "You have unsaved work. You will lose them if you open another file. Are you sure you want to continue?")
            if r == 'no':
                return
        
        file = tkFileDialog.askopenfile(mode="rb", **self.fileOptions)
        if not file == None:
            self.detachMetaController()
            
            for des in DataBoard.designers:
                des.destroy() 
                
            control.Utils.resetDataBoard()
            
            control.Utils.loadDesignFromFile(file)
            self.root.title(os.path.split(file.name)[1]+' - '+self.title)
            DataBoard.currentFile = file.name
            file.close()
            
            self.resetMenu()
            
            gui.Utils.modified()
            DataBoard.modified = False
            
            gui.Utils.reinitAllDesigners()
            gui.Utils.updateRegionEditors()
            gui.Utils.updateWaypointEditors()
    
    def startLogPlayback(self):
        file = tkFileDialog.askopenfile(**self.logOptions)
        if not file == None:
            thread = gui.LogPlayback.PlaybackThread(file)
            if len(thread.loglines) == 0:
                tkMessageBox.showerror('Log Playback Error', 'The file you specified is either empty, or not a correctly formatted gssp log file.')
                return
            DataBoard.logPlaybackThread = thread
            # PlaybackThread needs to be initialized and put into 
            # DataBoard before the next line.
            DataBoard.playbackControllerHandle = PlaybackController(self.root, os.path.split(file.name)[1])
            thread.setDaemon(True)
            # The playback thread should only be started after the playback
            # controller has been initialized and put into DataBoard
            thread.start()
            self.runMenu.entryconfigure(2, state=DISABLED)
            self.runMenu.entryconfigure(3, state=DISABLED)
            self.runMenu.entryconfigure(4, state=DISABLED)
            self.runMenu.entryconfigure(5, state=DISABLED)
            self.runMenu.entryconfigure(6, state=NORMAL)
            
    def stopLogPlayback(self):
        DataBoard.logPlaybackThread.started = False
        DataBoard.logPlaybackThread = None
        self.runMenu.entryconfigure(2, state=NORMAL)
        self.runMenu.entryconfigure(3, state=DISABLED)
        self.runMenu.entryconfigure(4, state=DISABLED)
        self.runMenu.entryconfigure(5, state=NORMAL)
        self.runMenu.entryconfigure(6, state=DISABLED)
        
    def resetMenu(self):
        for i in range(3,5):
            self.fileMenu.entryconfig(i, state=NORMAL)
                
        for i in range(1,3):
            self.editMenu.entryconfig(i, state=DISABLED)
        
        for i in range(1,6):
            self.designMenu.entryconfig(i, state=NORMAL)
        
        self.runMenu.entryconfig(1, state=NORMAL)
        self.runMenu.entryconfig(2, state=NORMAL)
        self.runMenu.entryconfig(5, state=NORMAL)

        self.menuButtonSave.configure(state=NORMAL)
        self.menuButtonSaveAs.configure(state=NORMAL)
        
    def newDesigner1D(self):
        
        d = DesignerSettings.DesignerSettings1D(self.root)
        if d.result == None:
            return
        designer = gui.DesignerTK.OneD.Designer1D(self.root, d.result)
        DataBoard.designers.append(designer)
    
    def newDesigner2D(self):
        d = DesignerSettings.DesignerSettings2D(self.root)
        if d.result == None:
            return
        designer = gui.DesignerTK.TwoD.Designer2D(self.root, d.result)
        DataBoard.designers.append(designer)
    
    def newRegTextEditor(self, geoX = None, geoY = None):
        if len(DataBoard.regEditors) > 0:
            DataBoard.regEditors[0].focus()
            DataBoard.regEditors[0].lift()
        else:
            editor = gui.TextEditor.RegionEditor.EditorFrame(self.root, geoX, geoY)
            DataBoard.regEditors.append(editor)
  
    def newWptTextEditor(self, geoX = None, geoY = None):
        if len(DataBoard.wptEditors) > 0:
            DataBoard.wptEditors[0].focus()
            DataBoard.wptEditors[0].lift()
        else:
            editor = gui.TextEditor.WaypointEditor.EditorFrame(self.root, geoX, geoY)
            DataBoard.wptEditors.append(editor)
    
    def newBooleanRegionEditor(self):
        if len(DataBoard.boolEditors) > 0:
            DataBoard.boolEditors[0].focus()
            DataBoard.boolEditors[0].lift()
        else:
            editor = gui.TextEditor.BooleanEditor.EditorFrame(self.root)
            DataBoard.boolEditors.append(editor)
    
    def startMetaController(self):
        d = MetaController.MetaController(self.root)
    
    def attachMetaController(self):
        d = AttachMetaController.AttachMetaController(self.root)
    
    def detachMetaController(self):
        gui.Utils.detachMetaController()
        gui.Utils.removeRobotImage()
        self.runMenu.entryconfig(2, state=NORMAL)
        self.runMenu.entryconfig(3, state=DISABLED)
        self.runMenu.entryconfig(4, state=DISABLED)
        self.runMenu.entryconfig(5, state=NORMAL)
        self.runMenu.entryconfig(6, state=DISABLED)
    
    def stopMetaController(self):
        d = tkMessageBox.askquestion("Stop Meta-controller", "This will only stop the meta-controller instance that you are currently attached to. Are you sure you want to continue?")
        if d == 'no':
            return
        if not DataBoard.asyncClient.write("exit"):
            tkMessageBox.showerror("Error", "Could not stop meta-controller. Something bad happened to the connection.")
            return
        self.detachMetaController()
    
    def run(self):
        #DataBoard.mainWindowHandle = self
        self.createWindow()
        self.createMenu()
        try:
            mainloop()
        except KeyboardInterrupt:
            pass
