'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on Dec 27, 2010

@author: Jimmy Li
'''

from control.DataBoard import DataBoard
from control.StateBoard import StateBoard
import control.WaypointController
import control.Region
import control.Waypoint
import control.TCPController
import time
import sys
import traceback
from config.Robots import Robots

class Environment():
    '''
    Code blocks are run by calling the runCodeBlock function of this class.
    CodeBlock and controllers (such as WaypointController) use this class.
    '''
    
    def __init__(self, region, owner, ownerName):
        '''
        region is the region that is the origin of this code block. For example,
        if region A instantiates a waypoint controller thread, which contains the
        waypoint that has this code block, then region A is the region here.
        owner is the TaskThread object that instantiated this object.
        
        ownerName is used to assist debugging when an error occurs in the code
        block. ownerName should say something like: 'Region r2' or 'Waypt Set w1'
        '''
        self.region_ = region
        self.owner = owner
        self.ownerName = ownerName
        self.alwaysBlock = True
    
    """
    functions or attributes that end with underscore can be called within
    code blocks, and is part of the runtime environment
    """
    
    def setBlocking_(self, blocking):
        if blocking == True:
            self.alwaysBlock = True
        else:
            self.alwaysBlock = False
    
    def goTo_(self, position = {}, name='', blocking=None):
        # Create a waypoint set with a single waypoint
        if blocking == None:
            blocking = self.alwaysBlock
        
        self.owner.lock.acquire()
        #This line accesses DataBoard. So it's best to lock thread.
        
        newWaypointSet = control.Waypoint.WaypointSet()
        
        if name == '':
            newWaypointSet.name = 'goTo('+newWaypointSet.id+') '+str(position)
        else:
            newWaypointSet.name = name
            
        self.owner.lock.release()
        
        waypoint = control.Waypoint.Waypoint()
        waypoint.position = position
        
        newWaypointSet.waypoints.append(waypoint)
        wptCont = control.WaypointController.TaskThread(self.region_, newWaypointSet)
        StateBoard.taskThreads.append(wptCont)
        wptCont.setDaemon(True)
        wptCont.start()
        if blocking:
            #Block and wait for task thread wptCont to finish
            while (wptCont.is_alive()):
                time.sleep(0.1)
    '''
    def set_(self, key, value):
        self.owner.lock.acquire()
        if value == None:
            del self.owner.reqStateChange[key]
        else:
            self.owner.reqStateChange[key] = value
        self.owner.lock.release()
    '''
                
    def startWaypoint_(self, wptSetName, blocking=None):
        if blocking == None:
            blocking = self.alwaysBlock
        for wayptID in DataBoard.waypointSets:
            if DataBoard.waypointSets[wayptID].name == wptSetName:
                wptSet = DataBoard.waypointSets[wayptID]
        wptCont = control.WaypointController.TaskThread(self.region_, wptSet)
        StateBoard.taskThreads.append(wptCont)
        wptCont.setDaemon(True)
        wptCont.start()
        if blocking:
            #Block and wait for task thread wptCont to finish
            while (wptCont.is_alive()):
                time.sleep(0.1)
    
    def startTCP_(self, port, blocking=None):
        if blocking == None:
            blocking = self.alwaysBlock
        tcpCont = control.TCPController.TaskThread(self.region_, port)
        StateBoard.taskThreads.append(tcpCont)
        tcpCont.setDaemon(True)
        tcpCont.start()
        if blocking:
            #Block and wait for task thread to finish
            while (tcpCont.is_alive()):
                time.sleep(0.1)
    
    def createRegion_(self,name=None,constraints=None,priority=None,command=None):
        #TODO
        pass
    
    def createWptSet_(self,name,wpts):
        #TODO
        pass
    
    def cancelGoTo_(self, name):
        self.cancelWaypoint_(name)
                
    def cancelWaypoint_(self, name):
        self.owner.lock.acquire()
        for taskThread in StateBoard.taskThreads:
            if taskThread.description == name + ' wpt controller thread':
                taskThread.kill = True
        self.owner.lock.release()
                
    def runCodeBlock(self, commands):
        # Set up namespace for the runtime environment
        region = self.region_
        goTo = self.goTo_
        startWaypoint = self.startWaypoint_
        cancelGoTo = self.cancelGoTo_
        cancelWaypoint = self.cancelWaypoint_
        startTCP = self.startTCP_
        sleep = time.sleep
        lock = self.owner.lock
        globals = StateBoard.globals
        setBlocking = self.setBlocking_
        state = StateBoard.currentState
        stateVars = Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars']
        
        # execute commands
        #print 'Executing', commands
        try:
            exec(commands)
        except:
            t, v, tb = sys.exc_info()
            errstr = "%s (%s %s)" % (t, v, traceback.print_tb(tb))
            print errstr
            errstr = errstr.replace("\n", '\n')
            DataBoard.metaControllerHandle.log('CODE', 'Error executing code of '+self.ownerName+': '+errstr)
'''
reg 1
x: -59.641830, -59.641387
y: 13.191735, 13.192157

reg 2
x: -59.641899, -59.641477
y: 13.191162, 13.191535


wpts
-59.641505, 13.191834
-59.641641, 13.191376
-59.641247, 13.191320
-59.641093, 13.191303
-59.640972, 13.191296
-59.641116, 13.191806

lock.acquire()

# Create region r1
r1 = control.Region.Region('r1')

r1.setConstraint({'Position X' : [-59.641830, -59.641387], 'Position Y': [13.191735, 13.192157]})

r1.commands = "seedSpreader('w', blocking=True)"

r1.priority = 30

DataBoard.regions.addRegion(r1)


# Create region r2

r2 = control.Region.Region('r2')

r2.setConstraint({'Position X' : [-59.641899, -59.641477], 'Position Y': [13.191162, 13.191535]})

r2.commands = "seedSpreader('e', blocking=True)"

r2.priority = 30

DataBoard.regions.addRegion(r2)

#Create waypoints

wpts = control.Waypoint.WaypointSet()

wpts.name = 'w1'

wpts.waypoints = [

#Enter first seedspreader reg
control.Waypoint.Waypoint(-59.641505, 13.191834, 0.0),

#Enter second seedspreader reg
control.Waypoint.Waypoint(-59.641641, 13.191376, 0.0),

# Follow reef edge
control.Waypoint.Waypoint(-59.641247, 13.191320, 0.0),
control.Waypoint.Waypoint(-59.641093, 13.191303, 0.0),
control.Waypoint.Waypoint(-59.640972, 13.191296, 0.0),

# Come back home
control.Waypoint.Waypoint(-59.641116, 13.191806, 0.0)
]

DataBoard.waypointSets[wpts.id] = wpts

lock.release()

startWaypoint('w1')
'''
