'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on 2010-06-01

@author: Jimmy Li
'''

from control.DataBoard import DataBoard
import control.Region
import pickle
import os
import datetime
import sys
import random

def getRandomColorHexCode():
    code = '#'
    for i in range(3):
        code += hex(random.randint(130, 240))[2:]
    return code

def getNewLogFilePath():
    pathname = os.path.dirname(sys.argv[0])
    abspathname = os.path.abspath(pathname)
    now = datetime.datetime.now()
    return os.path.join(abspathname, 'data', 'log_'+now.strftime("%Y-%m-%d_%H%M")+'.gssplog')

def primes(n): 
    """
    Prime number generator. Finds all prime numbers from 2 up to n (NOT generate n prime numbers)
    """
    if n==2: return [2]
    elif n<2: return []
    s=range(3,n+1,2)
    mroot = n ** 0.5
    half=(n+1)/2-1
    i=0
    m=3
    while m <= mroot:
        if s[i]:
            j=(m*m-3)/2
            s[j]=0
            while j<half:
                s[j]=0
                j+=m
        i=i+1
        m=2*i+3
    return [2]+[x for x in s if x]

def resetDataBoard():
    DataBoard.regionNumCounter = 1
    DataBoard.regions = control.Region.RegionHolder()
    DataBoard.booleanRegions = {}
    DataBoard.selectedRegion = ''        
    DataBoard.designers = []        
    DataBoard.waypointSetNumCounter = 1
    DataBoard.waypointSets = {} 
    DataBoard.selectedWaypointSet = ''
    DataBoard.selectedWaypoint = -1
    
    DataBoard.extEditorManager.files = {}
    DataBoard.history = []
    DataBoard.historyModifiedTime = 0
    DataBoard.modifiedTime = 0
    DataBoard.pointInHistory = 0
    DataBoard.currentFile = ''
    
    DataBoard.modified = False

def createSaveObject():
    obj = {'stateVarSet': DataBoard.currentStateVarSet,
           'wptSet': [], 
           'region': [], 
           'boolRegion': []}
    
        
    # Process waypoints
    for index in DataBoard.waypointSets:
        wptSet = DataBoard.waypointSets[index]
        wptSObj = {
            'name': wptSet.name,
            'color': wptSet.color,
            'waypoints': []
            }
        

        for waypt in wptSet.waypoints:
            wptSObj['waypoints'].append({
                    'commands': waypt.commands,
                    'position': waypt.position
                }
            )
        obj['wptSet'].append(wptSObj)
        
    # Process basic regions
    for reg in DataBoard.regions.getAllRegions():
        regObj = {
                  'constraints': reg.constraints,
                  'name': reg.name,
                  'color': reg.color,
                  'priority': reg.priority,
                  'bounded': reg.bounded,
                  'selection': reg.selection,
                  'repeat': reg.repeat,
                  'delay': reg.delay,
                  'commands': reg.commands
            }
        obj['region'].append(regObj)

    
    # Process boolean regions
    for index in DataBoard.booleanRegions:
        reg = DataBoard.booleanRegions[index]
        regObj = {}
        unionNames = []
        excluNames = []
        interNames = []
        for r in reg.union:
            unionNames.append(r.name)
        for r in reg.exclusion:
            excluNames.append(r.name)
        for r in reg.intersection:
            interNames.append(r.name)
        
        regObj = {
              'union': unionNames,
              'exclusion': excluNames,
              'intersection': interNames,
              'name': reg.name,
              'priority': reg.priority,
              'selection': reg.selection,
              'bounded': reg.bounded,
              'repeat': reg.repeat,
              'delay': reg.delay,
              'commands': reg.commands
            }
        obj['boolRegion'].append(regObj)
        
    return obj

def saveDesignToFile(file = None):
    if file == None:
        file = open(DataBoard.currentFile, "wb")
    obj = createSaveObject()
    pickle.dump(obj, file, pickle.HIGHEST_PROTOCOL)
    DataBoard.modified = False
    
def loadDesignFromFile(file):
    obj = pickle.load(file)
    DataBoard.currentStateVarSet = obj['stateVarSet']
    
    #DataBoard.waypointRadius = float(obj['options']['waypointRadius'])
    
    # Process waypoints
    for wptSObj in obj['wptSet']:
        
        wptSet = control.Waypoint.WaypointSet()
        wptSet.name = wptSObj['name']
        wptSet.color = wptSObj['color']
                
        for waypoint in wptSObj['waypoints']:
            wpt = control.Waypoint.Waypoint()
            wpt.position = waypoint['position']
            wpt.commands = waypoint['commands']
            wptSet.waypoints.append(wpt)
        
        DataBoard.waypointSets[wptSet.id] = wptSet
    
    
    # Process basic regions
    for regObj in obj['region']:
        reg = control.Region.Region()
        reg.name = regObj['name']
        reg.priority = regObj['priority']
        reg.color = regObj['color']
        reg.selection = regObj['selection']
        reg.bounded = regObj['bounded']
        reg.repeat = regObj['repeat']
        reg.delay = regObj['delay']
        reg.commands = regObj['commands']
        reg.constraints = regObj['constraints']
        DataBoard.regions.addRegion(reg)
     
    # Process boolean regions
    for regObj in obj['boolRegion']:
        reg = control.Region.BooleanRegion()
        reg.name = regObj['name']
        reg.priority = regObj['priority']
        reg.selection = regObj['selection']
        reg.bounded = regObj['bounded']
        reg.repeat = regObj['repeat']
        reg.delay = regObj['delay']
        reg.commands = regObj['commands']
        
        for item in regObj['union']:
            reg.union.append(DataBoard.regions.getRegionByName(item))
        
        for item in regObj['exclusion']:
            reg.exclusion.append(DataBoard.regions.getRegionByName(item))
        
        for item in regObj['intersection']:
            reg.intersection.append(DataBoard.regions.getRegionByName(item))
    
        DataBoard.booleanRegions[reg.id] = reg

'''
def loadWaypointsFromFile(file):
    wptSet = control.Waypoint.WaypointSet()
    while 1:
        rawline = file.readline() 
        line = rawline.strip()
        if line == '' or rawline == '':
            if len(wptSet.waypoints) != 0:
                newWptSet = control.Waypoint.WaypointSet()
                newWptSet.name = wptSet.name
                for wpt in wptSet.waypoints:
                    newWpt = control.Waypoint.Waypoint(float(wpt.position['Position X']), float(wpt.position['Position Y']))
                    newWpt.commands = wpt.commands
                    newWptSet.waypoints.append(newWpt)
                DataBoard.waypointSets[newWptSet.id] = newWptSet
                
                wptSet = control.Waypoint.WaypointSet()
            if rawline == '': 
                break
        elif line[0:1] == '#':
            continue
        else:
            comma = line.find(',')
            if comma < 0:
                wptSet.name = line
            else:                
                item = line[0:comma].strip()
                if not isFloat(item):
                    wptSet.name = item
                else:
                    x = item   
                    item = line[comma+1:].strip()
                    if not isFloat(item):
                        wptSet.name = item
                    else: 
                        y = item
                        wpt = control.Waypoint.Waypoint(x,y)
                        wptSet.waypoints.append(wpt)
                        
    if not DataBoard.mainWindowHandle == None:
        gui.Utils.reinitAllDesigners()
        gui.Utils.updateRegionEditors()
        gui.Utils.updateWaypointEditors()
'''


def getLogEntryTime(line):
    firstOpenBrack = line.find('[')
    firstCloseBrack = line.find(']')
    timestr = line[firstOpenBrack+1:firstCloseBrack]
    return float(timestr)
    
def isFloat(s):
    try:
        float(s)
        return True
    except ValueError:
        return False
