'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on 2010-05-27

@author: Jimmy Li
'''

import control.DataBoard
#import gui.Utils
import control.Utils
import time

class Region():
    '''
    Describes a region. See DataBoard for an explanation of how this stuff works
    
    IMPORTANT!!!!!!!
    LOCK THREAD WHEN CREATING A REGION SINCE THIS CLASS ACCESSES
    SHARED DATA IN THE DATABOARD
    '''

    def __init__(self, name=None):
        self.type = 'basic'
        
        self.constraints = {}
        
        # Get unique ID
        self.number = control.DataBoard.DataBoard.regionNumCounter
        control.DataBoard.DataBoard.regionNumCounter += 1
        
        self.id = "region%d" % (self.number)
        
        if name == None :
            self.name = "r%d" % (self.number)
        else:
            self.name = name
        
        self.commands = ""
        
        self.priority = 25
        
        # 1 = inside, 2 = outside
        self.selection = 1
        
        self.bounded = True
                
        # number of times to repeat
        self.repeat = 1
        
        # If repeat, how often to repeat (in seconds)
        self.delay = 0
        
        # When the commands of the region were last executed, expressed in seconds since the epoch
        #self.lastExecution = 0
        
        self.color = control.Utils.getRandomColorHexCode()
        
        # Whether the robot is inside the region or not
        self.inside = False
        
        # Indicates whether the robot has been inside this region at some point
        self.activated = False
        
        self.childThreads = []
        
        # Temporary var. Code elsewhere can use this
        self.temp = {}
        
    def setConstraint(self, parameters):
        """
        The parameters is a dictionary containing the parameters we want to constrain
        
        Example:
        setConstraint({'Position X' : [20,50], 'Position Z' : [35,70], 'Battery Level' : [12, 20]})
        
        The above example will add 3 contraints to this region. Each element in the dictionary has
        a key corresponding to the parameter we want to constrain, and each element is a two element
        list indicating the range for the region. This list must only contain two elements.
        Assuming that no constraints have been added to this
        region yet, this will result in a 3 dimensional region. This function can be called
        unlimited times to set constraints. If a constraint has already been set, setting it again
        will overwrite the old value. Setting a constraint to None instead of giving it a two-element
        list will delete that constraint.
        """
        for param in parameters:
            #if not self.constraints.has_key(param):
            #    continue
            if parameters[param] == None and self.constraints.has_key(param):
                del self.constraints[param]
                continue
            if parameters[param] == None:
                continue
            if parameters[param][0] == 'inf' and parameters[param][1] == 'inf' and self.constraints.has_key(param):
                del self.constraints[param]
                continue
            if parameters[param][0] == 'inf' or parameters[param][1] == 'inf':
                pass
            else:
                if parameters[param][0] > parameters[param][1]:
                    temp = parameters[param][1]
                    parameters[param][1] = parameters[param][0]
                    parameters[param][0] = temp
            self.constraints[param] = parameters[param]
            
    def dumpConstraints(self):
        print '-------------------------'
        print 'Dumping constraints for region: ', self.name
        print self.constraints
        print '-------------------------'

class RegionHolder():
    """
    A container to hold regions. See DataBoard for an explanation of how this stuff works.
    
    NOTE: If you want to change the type or number of dimensions contrained by a region changes,
    you must follow the following steps:
    
    1. Remove the region from the RegionHolder object
    2. Update the region
    3. Reenter the region into the RegionHolder
    
    This will make sure the region is put into the right location in the regions dictionary 
    """
    
    def __init__(self):
        self.regions = {}
        self.currPrimeIndex = 0
        self.primeMapping = {}
    
    def addRegion(self, region):
        key = self.calculateKey(region.constraints)
        if not self.regions.has_key(key):
            self.regions[key] = []
        self.regions[key].append(region)
    
    def removeRegion(self, region):
        key = self.calculateKey(region.constraints)
        if not self.regions.has_key(key):
            return
        index = self.regions[key].index(region)
        del self.regions[key][index]
    
    def getReleventRegions(self, constraints):
        '''
        This function gets all the regions that constrain
        at least all the parameters given in constraints.
        Relevent regions could also constrain other parameters. 
        This means that the regions returned must have dimension 
        larger or equal to constraints.
        '''
        returnArray = []
        key = self.calculateKey(constraints)
        for index in self.regions:
            if index % key == 0:
                returnArray += self.regions[index]
        return returnArray
    
    def getAllRegions(self):
        regList = []
        for index in self.regions:
            for reg in self.regions[index]:
                regList.append(reg)
        return regList
    
    def getRegionByID(self, regionID):
        for reg in self.getAllRegions():
            if reg.id == regionID:
                return reg
        return None
    
    def getRegionByName(self, regName):
        for reg in self.getAllRegions():
            if reg.name == regName:
                return reg
        return None
    
    def calculateKey(self, constraints):
        key = 1
        for param in constraints:
            if not self.primeMapping.has_key(param):
                self.primeMapping[param] = self.getNextPrimeNumber()
            key *= self.primeMapping[param]
        return key
    
    def getNextPrimeNumber(self):
        primeNumber = control.DataBoard.DataBoard.primes[self.currPrimeIndex]
        self.currPrimeIndex += 1
        return primeNumber
    
    def dumpRegions(self):
        print '-------------------------'
        print 'Region dump: '
        print self.regions
        print '-------------------------'

class BooleanRegion():
    '''
    These regions don't go into RegionHolder object. Boolean regions are
    composed of Region objects.
    '''
    def __init__(self, name=None):
        
        self.type = 'boolean'
        
        self.union = []
        self.exclusion = []
        self.intersection = []
        
        # Get unique ID
        self.number = control.DataBoard.DataBoard.regionNumCounter
        control.DataBoard.DataBoard.regionNumCounter += 1
        
        self.id = "bregion%d" % (self.number)
        
        if name == None :
            self.name = "b%d" % (self.number)
        else:
            self.name = name
        
        self.commands = ""
        
        self.priority = 25
        
        # 1 = inside, 2 = outside
        self.selection = 1
        
        #1 = unbounded, 2 = bounded
        self.bounded = 1
        
        # 1 = run once, 2 = repeat
        self.repeat = 1
        
        self.delay = 0
        
        # When the commands of the region were last executed, expressed in seconds since the epoch
        #self.lastExecution = 0
        
        # Whether the robot is inside the region or not
        self.inside = False
        
        # Indicates whether the robot has been inside this region at some point
        self.activated = False
        
        self.childThreads = []
        
        # Temporary var. Code elsewhere can use this
        self.temp = {}
