'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on Dec 27, 2010

@author: Jimmy Li
'''

import threading
import time
import control.StateBoard
from control.DataBoard import DataBoard
import math
import control.Runtime

class TaskThread(threading.Thread):
    '''
    The code in each code block attached to a region is run
    in this thread. This allows for blocking statements: ie. startWaypoint('Waypoint Set 1', blocking=True),
    goto(10,13,blocking=True), or startTCP(9551, blocking=True)
    '''

    def __init__(self, region):
        '''
        Constructor
                
        commands contains the content
        of the code block . repeat indicates the number of times to run this
        code block. delay specifies the interval between consecutive executions.
        '''
        threading.Thread.__init__(self)
        
        self.lock = threading.RLock()
        
        self.region = region
        
        self.commands = region.commands
        self.repeat = region.repeat
        self.delay = region.delay
        
        self.priority = region.priority # inherit priority
        
        if self.repeat == 'inf':
            self.repeat = -1
                
        self.runTimeEnv = control.Runtime.Environment(region, self, 'Region ' + self.region.name) 
        
        self.reqStateChange = {}
        
        self.description = region.name + ' code block thread'
        
        self.alive = True
        
    def run(self):
        counter = 1
        while (1):
            if not control.StateBoard.StateBoard.metaControlStarted:
                break
            
            if not self.region.inside and self.region.bounded:
                '''
                The region is bounded and the robot is outside it. Since the region
                is a bounding region, not just a triggering region, this thread
                should be paused.
                '''
                self.reqStateChange.clear()
                time.sleep(0.1)
                continue
            
            if not int(self.repeat) < 0 and counter > int(self.repeat):
                '''
                If repeat is not infinite, and the code block has already
                run for the specified number of times.
                '''
                break
            
            self.lock.acquire()
            DataBoard.metaControllerHandle.log('CODE', 'Executing code block of region ' + self.region.name + '; Repetition: '+ str(counter))
            self.lock.release()
            
            self.runTimeEnv.runCodeBlock(self.commands)
            
            counter+=1
            
            time.sleep(self.delay)
        
        self.lock.acquire()
        self.reqStateChange.clear()
        self.lock.release()
        
        self.lock.acquire()
        DataBoard.metaControllerHandle.log('REGION', 'Deactivating region ' + self.region.name + '; Total repetition: '+ str(counter-1))
        self.lock.release()
        '''
        Wait for robot to go outside the region
        '''
        while self.region.inside:
            time.sleep(0.1)
        
        self.region.activated = False
        
        '''
        End of the region's life cycle
        '''
        self.alive = False
        
        