'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on 2011-07-04

@author: Jimmy Li
'''

from Tkinter import *
import tkMessageBox
from control.DataBoard import DataBoard
import gui.Utils
import datetime

class PlaybackController(Toplevel):
    '''
    classdocs
    '''

    def __init__(self, master, fileName):
        '''
        Constructor
        '''
        Toplevel.__init__(self, master, background="#F0F0F0")
        
        self.master = master
        self.title("Log File Playback Controller")
        self.protocol("WM_DELETE_WINDOW", self.closeWindow)
        self.resizable(width=TRUE, height=TRUE)
        self.geometry("+%d+%d" % (master.winfo_rootx()+50,
                                  master.winfo_rooty()+50))
        
        #self.tk.call('wm', 'iconphoto', self._w, DataBoard.icon)
        
        self.bind("<Control-KeyRelease-z>", gui.Utils.undo)
        self.bind("<Control-KeyRelease-r>", gui.Utils.redo)
        
        playbackLabel = Label(self, background="#F0F0F0")
        playbackLabel.grid(row=0, column=0)
        
        self.sliderDrag = False
        self.playbackSlider = Scale(self, background="#F0F0F0", command=self.sliderMoved, from_=0, to=0, orient=HORIZONTAL, length=500, label='00:00:00', showvalue=False)
        self.playbackSlider.grid(row=1, column=0, sticky=(E,W), pady=5)
        self.playbackSlider.bind("<Button-1>", self.initSliderDrag)
        self.playbackSlider.bind("<ButtonRelease-1>", self.playbackSliderDragged)
        self.columnconfigure(0, weight=1)
        
        buttonFrame = Frame(self)
        buttonFrame.grid(row=2,column=0)
        self.playPauseButton = Button(buttonFrame, text='Pause', command=self.togglePausePlay)
        self.playPauseButton.pack(side=LEFT)
        Button(buttonFrame, text='Close Playback', command=self.closeWindow).pack(side=LEFT)
        
        self.playbackThread = DataBoard.logPlaybackThread
        if len(self.playbackThread.loglines) == 0:
            tkMessageBox.showerror('Error', 'Log file is empty')
            self.closeWindow()
        else:
            self.startTime = int(self.playbackThread.loglines[0]['time'])
            self.numLogLines = len(self.playbackThread.loglines)
            self.stopTime = int(self.playbackThread.loglines[self.numLogLines-1]['time'])
            elapsed = self.stopTime - self.startTime
            strElapsed = str(datetime.timedelta(seconds=elapsed))
            playbackLabel.configure(text='Playing back: '+fileName+'; Length: '+strElapsed)
            self.playbackSlider.configure(to=elapsed)
    
    def togglePausePlay(self):
        if self.playbackThread.paused == False:
            self.playbackThread.paused = True
            self.playPauseButton.configure(text='Play')
        else:
            self.playbackThread.paused = False
            self.playPauseButton.configure(text='Pause')
    
    def initSliderDrag(self, event=None):
        self.sliderDrag = True
    
    def sliderMoved(self, event=None):
        '''
        Update time display.
        '''
        val = int(self.playbackSlider.get())
        self.playbackSlider.configure(label=str(datetime.timedelta(seconds=int(val)))) 
    
    def playbackSliderDragged(self, event=None):
        '''
        slider movement resulting from user drag
        '''
        self.sliderDrag = False
        if not self.playbackThread.is_alive():
            return
        val = int(self.playbackSlider.get())
        self.playbackThread.jumpToTime(val)
    
    def updateSliderPosition(self, time):
        '''
        slider movement resulting from playback, not user drag.
        called by the playback thread
        '''
        if self.sliderDrag:
            return
        delta = time - self.startTime
        self.playbackSlider.set(delta) 
    
    def closeWindow(self):
        DataBoard.mainWindowHandle.stopLogPlayback()
        self.destroy()
        