'''
Created on 2011-06-22

@author: Jimmy Li
'''

import math
import threading

import roslib; roslib.load_manifest('gsspmc')
import rospy
from turtlesim.msg import Pose, Velocity

class Driver():
    '''
    A GSSP driver for ROS's turtlesim simulator
    '''

    def __init__(self):
        self.lock = threading.Lock()
        self.currentState = {}
        
        rospy.init_node('gsspmc')
        self.pub = rospy.Publisher('turtle1/command_velocity', Velocity)
        rospy.Subscriber("turtle1/pose", Pose, self.callback)
        
        self.velocity = 3.0
        self.yawrate = 5.0
        
    def callback(self, data):
        self.lock.acquire()
        self.currentState['Position X'] = data.x
        self.currentState['Position Y'] = data.y
        self.currentState['Angle'] = data.theta
        self.currentState['Velocity'] = data.linear_velocity
        self.currentState['Yaw'] = data.angular_velocity
        self.lock.release()

    
    def getState(self):
        '''
        Return current state of the robot.
        '''
        return self.currentState
    
    def changeState(self, stateChanges):
        '''
        Change the robot's state as specified by stateChanges.
        '''       
        if len(stateChanges.keys()) == 0:
            self.pub.publish(Velocity(linear=0.0, angular=0.0))
        
        if stateChanges.has_key('Velocity'):
            self.velocity = stateChanges['Velocity']
            self.pub.publish(Velocity(linear=self.velocity, angular=self.currentState['Yaw']))

        if stateChanges.has_key('Position X') or stateChanges.has_key('Position Y'):
            toX = 0
            toY = 0
            
            # If only one of Position X or Position Y is specified, we'll use the current value

            if not stateChanges.has_key('Position X'):
                toX = self.currentState['Position X']
            else:
                toX = stateChanges['Position X']
                
            if not stateChanges.has_key('Position Y'):
                toY = self.currentState['Position Y']
            else:
                toY = stateChanges['Position Y']
            
            self._moveTo(toX, toY)
            
    
    def _moveTo(self, toX, toY):
        '''
        Adjust the linear and angular velocity such that the robot
        moves to the point (toX, toY)
        '''
        px = self.currentState['Position X']
        py = self.currentState['Position Y']
        pa = self.currentState['Angle']
        
        
        dx = toX - px
        dy = toY - py
        
        # shift the origin to the position of the robot
        
        newToX = toX - px
        newToY = toY - py
        
        # get the angle of the destination vector
        
        angle = 0
        
        if newToX == 0 and newToY == 0:
            # robot is on the destination. Do nothing
            pass
        elif dy == 0:
            if newToX > 0:
                angle = 0
            else:
                angle = math.pi
        elif dx == 0:
            if newToY > 0:
                angle = math.pi/2
            else:
                angle = math.pi * 3 / 2
        else:
            angle = math.atan(math.fabs(newToX)/math.fabs(newToY))
            if newToX > 0 and newToY > 0:
                angle = math.pi/2 - angle 
            elif newToX < 0 and newToY > 0:
                angle = math.pi/2 + angle
            elif newToX < 0 and newToY < 0:
                angle = math.pi * 3 / 2 - angle
            else:
                angle = math.pi * 3 / 2 + angle
      
        if pa < 0:
            pa = math.pi * 2 + pa
        
        if math.fabs(angle - pa) < 0.1:
            yr = 0
        else:
            if angle > pa:
                # turn clock wise
                dangle = angle - pa
                if dangle >= math.pi:
                    yr = math.copysign(self.yawrate, -1)
                else:
                    yr = self.yawrate
            else:
                dangle = pa - angle
                if dangle >= math.pi:
                    yr = self.yawrate
                else:
                    yr = math.copysign(self.yawrate, -1)
        
        self.pub.publish(Velocity(linear=self.velocity, angular=yr))
        
    def cleanup(self):
        pass
    
