'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on 2010-07-19

@author: Jimmy Li
'''

from control.StateBoard import StateBoard
from control.DataBoard import DataBoard
import control.RegCodeBlock
import threading
import time
import heapq
import os
import control.AsyncServer
import socket, asyncore, asynchat
import copy
import sys
import traceback

class MetaController(threading.Thread):
    def __init__(self, driverName, comport, logfilePath):
        self.driverName = driverName
        self.comport = comport
        self.logfilePath = logfilePath
        
        threading.Thread.__init__(self)
        self.lock = threading.RLock()
        StateBoard.currentStateVarSet = DataBoard.currentStateVarSet
        driver = __import__('driver.'+os.path.splitext(driverName)[0], globals(), locals(), ['Driver'], -1)
        self.robotDriver = driver.Driver()
        
        StateBoard.currentRobotDriver = self.robotDriver
        StateBoard.previousStateChanges.clear()
        self.initRegions()
        
        self.logfile = open(logfilePath, "a")
        
        self.server = control.AsyncServer.Server(port=comport)
        DataBoard.asyncServer = self.server
        self.server_thread = threading.Thread(target = asyncore.loop)
        self.server_thread.setDaemon(True)
        self.server_thread.start()
        
        DataBoard.metaControllerHandle = self
        
    
    def initRegions(self):
        for reg in DataBoard.regions.getAllRegions():
            reg.inside = False
            reg.activated = False
        
        for index in DataBoard.booleanRegions:
            reg = DataBoard.booleanRegions[index]
            reg.inside = False
            reg.activated = False        
    
    '''
    def sendPickled(self, msgtype, struct):
        msg = 
        log = '['+str(time.time())+'] '+msgtype+': '+msg+"\n"
    '''
    
    def log(self, msgtype, msg, send=True):
        log = '['+str(time.time())+'] '+msgtype+': '+msg+"\n"
        try:
            self.logfile.write(log)  
            self.logfile.flush()          
        except ValueError:
            pass
        if send:
            try:
                DataBoard.asyncServer.write(log)
            except:
                t, v, tb = sys.exc_info()
                print "An error occurred with the connection to the gui: Caught exception: %s (%s %s)" % (t, v, traceback.print_tb(tb))
                
    def run(self):
        StateBoard.metaControlStarted = True
        self.log('META-CONTROL', 'Meta-controller thread started.')
        self.log('META-CONTROL', 'Listening to port ' + str(self.comport))
        self.log('META-CONTROL', 'Using log file '+self.logfilePath)
        while(1):
            self.lock.acquire()
            
            ##########################################################
            # Check to see if meta-controller should terminate #######
            ##########################################################
            if not StateBoard.metaControlStarted:
                self.server.handle_close()
                self.logfile.close()
                self.robotDriver.cleanup()
                print 'Meta-controller cleanup finished.'
                break
            
            
            ##########################################################
            # Clear the state changes from the last iteration ########
            ##########################################################
            StateBoard.stateChanges.clear()
            
            
            ##########################################################
            # Get the current state ##################################
            ##########################################################
            newState = copy.deepcopy(self.robotDriver.getState())
            
            # Logging for state is differential
            for state in newState:
                if not StateBoard.currentState.has_key(state) or not StateBoard.currentState[state] == newState[state]: 
                    #print 'logging'
                    self.log('STATE', repr(newState))
                    #self.sendPickled('STATE', newState)
                    break
            
            StateBoard.currentState = newState
            
            
            ##########################################################
            # Find all the regions that the robot is in ##############
            ##########################################################
            insideRegions = []
            
            # Process basic regions
            for reg in DataBoard.regions.getAllRegions():
                satisfied = True
                for cons in reg.constraints:

                    # Check for the robot being outside the region
                    if reg.constraints[cons][0] == 'inf' and reg.constraints[cons][1] == 'inf':
                        continue
                    elif not StateBoard.currentState.has_key(cons):
                        satisfied = False
                        break
                    elif reg.constraints[cons][0] == 'inf': # Lower bound is minus infinity
                        if StateBoard.currentState[cons] > reg.constraints[cons][1]:
                            satisfied = False
                            break
                    elif reg.constraints[cons][1] == 'inf': # Upper bound is plus infinity
                        if StateBoard.currentState[cons] < reg.constraints[cons][0]:
                            satisfied = False
                            break
                    else:
                        if reg.constraints[cons][0] > StateBoard.currentState[cons] or reg.constraints[cons][1] < StateBoard.currentState[cons]:
                            satisfied = False
                            break
                        
                if satisfied:
                    if reg.selection == 1:
                        reg.inside = True
                    else:
                        reg.inside = False
                else:
                    if reg.selection == 2:
                        reg.inside = True
                    else:
                        reg.inside = False
                
                if reg.inside:
                    insideRegions.append(reg)
                
            # Process boolean regions
            for index in DataBoard.booleanRegions:
                reg = DataBoard.booleanRegions[index]
                union_satisfied = False
                exclu_satisfied = True
                inter_satisfied = True
                satisfied = False
                if not len(reg.union) > 0:
                    union_satisfied = True
                else:
                    for r in reg.union:
                        if r.inside:
                            union_satisfied = True
                            break
                for r in reg.exclusion:
                    if r.inside:
                        exclu_satisfied = False
                        break
                for r in reg.intersection:
                    if not r.inside:
                        inter_satisfied = False
                        break
                
                if union_satisfied and exclu_satisfied and inter_satisfied:
                    if reg.selection == 1:
                        satisfied = True
                else:
                    if reg.selection == 2:
                        satisfied = True
                
                if satisfied:
                    reg.inside = True
                else:
                    reg.inside = False
                
                if reg.inside:
                    insideRegions.append(reg)
            
            
            ##########################################################
            # Activate regions the robot is in #######################
            ##########################################################
            for region in insideRegions:
                if not region.activated:
                    # Activate any regions that has not been activated. Thread is currently locked
                    # so the codeThread shouldn't start running just yet
                    self.log('REGION', 'Activating ' + region.name)
                    codeThread = control.RegCodeBlock.TaskThread(region)
                    StateBoard.taskThreads.append(codeThread)
                    codeThread.setDaemon(True)
                    codeThread.start()
                    region.activated = True
            
            
            ##########################################################
            # Process Task Threads to determine state changes ########
            ##########################################################
                        
            # Build a priority queue based on the priority of task threads.
            # Given that two threads have the same priority,
            # the one started later will have higher priority.
            priorityQueue = []
            
            # StateBoard.taskThreads will be rebuilt, weaning out
            # dead threads
            newTaskThreads = []
                        
            # counter is a tie-breaking mechanism for threads with the same priority.
            # See python documentation for heapq
            counter = 0
            
            # Note that tasks are appended to StateBoard.taskThreads
            # when they are started. So, the order or task threads in
            # StateBoard.taskThreads tells us the order in which the threads
            # are started.
            for taskThread in StateBoard.taskThreads:
                # taskThread.is_alive() is not reliable because it is true
                # if the thread haven't even started.
                if taskThread.alive:
                    newTaskThreads.append(taskThread)
                    heapq.heappush(priorityQueue, [taskThread.priority, counter, taskThread])
                counter += 1
                        
            # Build state changes. Higher priority threads get
            # to overwrite the changes requested by lower priority
            # threads if conflict arises.
            while len(priorityQueue) > 0:
                entry = heapq.heappop(priorityQueue)
                taskThread = entry[2]
                for name, value in taskThread.reqStateChange.items():
                    StateBoard.stateChanges[name] = value
            
            # Update StateBoard.taskThreads
            StateBoard.taskThreads = newTaskThreads
            
            
            ##########################################################
            # Send state changes to robot driver #####################
            ##########################################################
            
            # Logging for state changes is differential
            for state in StateBoard.stateChanges:
                if not StateBoard.previousStateChanges.has_key(state) or not StateBoard.previousStateChanges[state] == StateBoard.stateChanges[state]: 
                    self.log('DRIVER', 'Request state change ' + repr(StateBoard.stateChanges))
                    break
            
            # Convert the state changes to robot-specific commands
            self.robotDriver.changeState(StateBoard.stateChanges)
            
            # Update StateBoard.previousStateChanges to be used during the next iteration
            StateBoard.previousStateChanges = copy.deepcopy(StateBoard.stateChanges)
            
            
            self.lock.release()
            time.sleep(0.1)
        
