'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on Dec 30, 2010

@author: Jimmy Li, Anqi Xu
'''

import socket, asyncore, asynchat
from control.StateBoard import StateBoard

class ClientHandler(asynchat.async_chat):
  def __init__(self, sock, addr):
    self.alive = False
    asynchat.async_chat.__init__(self, sock=sock)
    self.addr = addr
    self.input_buffer = ''
    self.set_terminator('\n')
    self.alive = True

  def close(self):
    self.alive = False
    asynchat.async_chat.close(self)

  def close_when_done(self):
    self.alive = False
    asynchat.async_chat.close_when_done(self)

  def collect_incoming_data(self, data):
    self.input_buffer = self.input_buffer + data

  def found_terminator(self):
    #print '\n' + repr(self.addr) + '> [' + self.input_buffer + '] (' + str(len(self.input_buffer)) + ')'
    if self.input_buffer.strip() == 'exit':
        print 'A client has asked me to terminate.'
        StateBoard.metaControlStarted = False
    self.input_buffer = ''

  def write(self, data):
    if self.alive:
      self.push(data)


class Server(asyncore.dispatcher):
  def __init__(self, host = "", port = 5007, maxNumClients = 5): # TODO: find out why host = "" works but host = "localhost" doesn't? also what if we pass host = socket.getHomeName()?
    self.alive = False
    asyncore.dispatcher.__init__(self)
    self.maxNumClients = maxNumClients
    self.create_socket(socket.AF_INET, socket.SOCK_STREAM)
    self.set_reuse_addr()
    self.bind((host, port))
    self.listen(5) # Allow up to N clients to WAIT for connection
    self.alive = True
    self.clients = []

  def close(self):
    self.alive = False
    for client in self.clients:
      client.close()
    del self.clients[:]
    asyncore.dispatcher.close(self)

  def handle_close(self):
    self.close()

  def prune_dead_clients(self):
    i = 0
    while i < len(self.clients):
      if not self.clients[i].alive:
        self.clients[i].close()
        del self.clients[i]
      else:
        i += 1

  def handle_accept(self):
    self.prune_dead_clients()
    pair = self.accept()
    if pair is None:
      pass # Apparently asyncore's accept can fail at times
    else:
      sock, addr = pair
      if len(self.clients) < self.maxNumClients:
        #print '\nIncoming connection from %s' % repr(addr)
        handler = ClientHandler(sock, addr)
        self.clients.append(handler)
      else: # Do not accept more than maxNumClients clients at once
        sock.shutdown(socket.SHUT_RDWR)
        sock.close()

  def write(self, data): # Automatically prunes dead clients at the same time
    if not data or not self.alive:
      return False
    #print 'sending data: ', data
    if data[-1] != '\n':
      data = data + '\n'
    i = 0
    written = False
    while i < len(self.clients):
      if not self.clients[i].alive:
        self.clients[i].close()
        del self.clients[i]
      else:
        self.clients[i].write(data)
        written = True
        i += 1
    return written

'''
# Main
server = Server()
server_thread = threading.Thread(target = asyncore.loop)
server_thread.setDaemon(True)
server_thread.start()
while server.alive:
  data = raw_input('\nType text (Enter q or Q to Quit):')
  if not data:
    continue
  elif data != 'Q' and data != 'q':
    if not server.write(data):
      print '\nWrite failed: no clients'
  else:
    break
server.handle_close()
server_thread.join(0.5)
'''
