'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on 2011-06-09

@author: Jimmy Li
'''

from Tkinter import *
import gui.Utils
import gui.dialogs.EditConstraint
from control.DataBoard import DataBoard
from config.Robots import Robots
import tkMessageBox

import time

class EditorFrame(Toplevel):
    '''
    classdocs
    '''
    
    def __init__(self, master, geoX = None, geoY = None):
        '''
        Constructor
        '''
        Toplevel.__init__(self, master)
        
        self.master = master
        self.title("Region Text Editor")
        self.listBoxWidth = 12
        self.listBoxHeight = 27 
        
        self.regCmdWidth = 40 
        self.regCmdHeight = 24
        self.regCstWidth = 27
        self.regCstHeight = 13
        self.standardOptionWidth = 19
        self.launchExtEditorEntryWidth = 20
        
        self.protocol("WM_DELETE_WINDOW", self.closeWindow)
        
        self.resizable(width=TRUE, height=TRUE)
        
        if geoX == None:
            geoX = master.winfo_rootx()
        
        if geoY == None:
            geoY = master.winfo_rooty()+20
            
        self.geometry("+%d+%d" % (geoX,geoY))
        
        #self.tk.call('wm', 'iconphoto', self._w, DataBoard.icon)
        
        self.bind("<Control-KeyRelease-z>", gui.Utils.undo)
        self.bind("<Control-KeyRelease-r>", gui.Utils.redo)
        
        self.editingRegion = None
        self.designerSelectionHighlight = "#FF9966"
        self.selectionHighlight = "#A8D4FF"
        self.selectionTextColor = "black"
        self.constraintNames = []
        
        # REGION EDITOR MASTER FRAME
        regionMasterFrame = Frame(self, background="#F0F0F0")
        regionMasterFrame.grid(row=0, column=0, sticky=(N,S,E,W), ipady=3, ipadx=5)
        self.columnconfigure(0, weight=1)
        self.rowconfigure(0, weight=1)
        
        # REGION LIST FRAME
        regionListFrame = Frame(regionMasterFrame)
        regionListFrame.grid(row=0, column=0, sticky=(N,S), padx=10, pady=5)
        regionMasterFrame.rowconfigure(0,weight=1)
        regionMasterFrame.columnconfigure(1,weight=1)
        
        scrollbar = Scrollbar(regionListFrame, orient=VERTICAL)
        self.regionListBox = Listbox(regionListFrame, selectforeground = self.selectionTextColor, selectbackground=self.selectionHighlight, exportselection=0, height=self.listBoxHeight, width=self.listBoxWidth,  selectmode=SINGLE, yscrollcommand=scrollbar.set)
        scrollbar.config(command=self.regionListBox.yview)
        scrollbar.pack(side=RIGHT, fill=Y)
        self.regionListBox.pack(side=TOP, fill=BOTH, expand=1)
        
        self.regionListBox.insert(END, "---- Regions ----")
        
        self.regionListBox.bind("<<ListboxSelect>>", self.clickRegionList)
        
        
        # REGION COMMAND EDITOR FRAME
        regionCmdFrame = Frame(regionMasterFrame, background="#F0F0F0")
        regionCmdFrame.grid(row=0, column=1, padx=5, pady=5, sticky=(N,S,E,W))
        regionCmdFrame.rowconfigure(1, weight=1)
        regionCmdFrame.columnconfigure(0, weight=1)
        
        # --- COMMANDS
        Label(regionCmdFrame, text="Code", background="#F0F0F0").grid(row=0, column=0, columnspan=2, sticky=W, padx=0)
        scrollbar = Scrollbar(regionCmdFrame)
        scrollbar.grid(row=1, column=1, sticky=(N,S), pady=5)
        self.regionCommandEditor = Text(regionCmdFrame, width=self.regCmdWidth, height=self.regCmdHeight, yscrollcommand=scrollbar.set)
        self.regionCommandEditor.grid(row=1, column=0, sticky=(N,S,E,W), padx=0, pady=5)        
        self.regionCommandEditor.config(state=DISABLED)
        scrollbar.config(command=self.regionCommandEditor.yview)
        self.regionCommandEditor.bind("<KeyRelease>", self.updateRegionCommand)
        
        # LAUNCH EXTERNAL EDITOR FRAME
        launchExtEditorFrame = Frame(regionCmdFrame, background="#F0F0F0")
        launchExtEditorFrame.grid(row=2, column=0, padx=0, pady=5, sticky=(E,W))
        launchExtEditorFrame.columnconfigure(0,weight=1)
        
        # --- LAUNCH EXTERNAL EDITOR
        self.extEditorEntry = Entry(launchExtEditorFrame, width = self.launchExtEditorEntryWidth)
        self.extEditorEntry.bind("<Return>", self.launchExtEditor)
        self.extEditorEntry.grid(row=0, column=0, sticky=(E,W))
        if DataBoard.rcData.has_key('text_editor'):
            self.extEditorEntry.insert(0,DataBoard.rcData['text_editor'])
        self.launchExtEditorButton = Button(launchExtEditorFrame, command=self.launchExtEditor)
        self.launchExtEditorButton.grid(row=0,column=1, padx=5)
        
        
        # REGION OPTIONS EDITOR FRAME
        regionEditorFrame = Frame(regionMasterFrame, background="#F0F0F0")
        regionEditorFrame.grid(row=0, column=2, padx=5, pady=5, sticky=N+S)
        regionEditorFrame.rowconfigure(8,weight=1)
        
        # ---- NAME
        Label(regionEditorFrame, text="Name", background="#F0F0F0").grid(row=1, column=0, sticky=W, padx=5, pady=3)
        self.regionNameEditor = Entry(regionEditorFrame, width = self.standardOptionWidth)
        self.regionNameEditor.grid(row=1, column=1, sticky=W, padx=5)
        self.regionNameEditor.config(state=DISABLED)
        self.regionNameEditor.bind("<KeyRelease>", self.updateRegionName)
        
        # ---- PRIORITY
        Label(regionEditorFrame, text="Priority", background="#F0F0F0").grid(row=2, column=0, sticky=W, padx=5, pady=3)
        self.regionPriorityEditor = Entry(regionEditorFrame, width = self.standardOptionWidth)
        self.regionPriorityEditor.grid(row=2, column=1, sticky=W, padx=5)
        self.regionPriorityEditor.config(state=DISABLED)
        self.regionPriorityEditor.bind("<KeyRelease>", self.updateRegionPriority)
        
        # ---- SELECTION
        Label(regionEditorFrame, text="Selection", background="#F0F0F0").grid(row=3, column=0, sticky=W, padx=5, pady=3)
        self.regionSelectionOptionText = ["Inside", "Outside"]
        self.regionSelectionVar = StringVar(regionEditorFrame)
        self.regionSelectionVar.set(self.regionSelectionOptionText[0])
        self.regionSelectionOption = OptionMenu(regionEditorFrame, self.regionSelectionVar, *tuple(self.regionSelectionOptionText), command=self.updateRegionSelection)
        self.regionSelectionOption.grid(sticky=W, row=3, column=1, pady=2)
        
        # ---- BOUNDING        
        Label(regionEditorFrame, text="Bounded", background="#F0F0F0").grid(row=4, column=0, sticky=W, padx=5, pady=3)
        self.regionBoundedOptionText = ["Yes", "No"]
        self.regionBoundedVar = StringVar(regionEditorFrame)
        self.regionBoundedVar.set(self.regionBoundedOptionText[0])
        self.regionBoundedOption = OptionMenu(regionEditorFrame, self.regionBoundedVar, *tuple(self.regionBoundedOptionText), command=self.updateRegionBounded)
        self.regionBoundedOption.grid(sticky=W, row=4, column=1, pady=2)
        
        # --- REPEAT AND DELAY
        Label(regionEditorFrame, text="Name", background="#F0F0F0").grid(row=1, column=0, sticky=W, padx=5, pady=3)
        self.regionNameEditor = Entry(regionEditorFrame, width = self.standardOptionWidth)
        self.regionNameEditor.grid(row=1, column=1, sticky=W, padx=5)
        self.regionNameEditor.config(state=DISABLED)
        self.regionNameEditor.bind("<KeyRelease>", self.updateRegionName)
        
        Label(regionEditorFrame, text="Repeat", background="#F0F0F0").grid(row=5, column=0, sticky=W, padx=5, pady=3)
        #  (ie: inf, 1, 25)
        self.regionRepeat = Entry(regionEditorFrame, width=self.standardOptionWidth)
        self.regionRepeat.grid(row=5, column=1, sticky=W, padx=5)
        self.regionRepeat.bind("<KeyRelease>", self.updateRegionRepeat)
                
        Label(regionEditorFrame, text="Delay", background="#F0F0F0").grid(row=6, column=0, sticky=W, padx=5, pady=3)
        self.regionDelay = Entry(regionEditorFrame, width=self.standardOptionWidth)
        self.regionDelay.bind("<KeyRelease>", self.updateRegionDelay)
        self.regionDelay.grid(row=6, column=1, sticky=W, padx=5)
                
        #-- CONSTRAINTS
        regionConstraintDisplayFrame = Frame(regionEditorFrame)
        regionConstraintDisplayFrame.grid(row=8, columnspan=2, column=0, sticky=(N,S), padx=4, pady=5)
        scrollbar2 = Scrollbar(regionConstraintDisplayFrame, orient=VERTICAL)
        self.regionConstraintDisplay = Listbox(regionConstraintDisplayFrame, height=self.regCstHeight, width=self.regCstWidth, yscrollcommand=scrollbar2.set, exportselection=0)
        scrollbar2.config(command=self.regionConstraintDisplay.yview)
        scrollbar2.pack(side=RIGHT, fill=Y)
        self.regionConstraintDisplay.pack(side=TOP, fill=BOTH, expand=1)
        self.regionConstraintDisplay.insert(END, "----------------- Constraints -----------------")
        self.regionConstraintDisplay.bind("<<ListboxSelect>>", self.clickConstraint)
        
        # ---- DELETING
        buttonFrame = Frame(regionEditorFrame)
        buttonFrame.grid(row=9, column=1, pady=5, padx=5, sticky=W)
        self.deleteRegionButton = Button(buttonFrame, text="Delete Region", command=self.deleteRegion)
        self.deleteRegionButton.pack(side=LEFT)
        self.deleteRegionButton.configure(state=DISABLED)
        
        self.refreshRegionList()
        self.updateSelectedRegion()
        
    def clickRegionList(self,event):
        sel = self.regionListBox.curselection()
        if len(sel) == 0:
            return
        index = sel[0]
        if index == '0':
            return
        self.editingRegion = self.regionListBoxData[int(index)-1]
        DataBoard.selectedRegion = self.editingRegion.id
        gui.Utils.redrawAllDesigners()
        gui.Utils.updateRegionEditors()
        self.updateSelectedRegion()
    
    def clickConstraint(self, event):
        if self.editingRegion == None:
            return
        if self.editingRegion.type == 'boolean':
            return
        sel = self.regionConstraintDisplay.curselection()
        if len(sel) == 0:
            return
        index = sel[0]
        if index == '0':
            return
        
        name = self.constraintNames[int(sel[0])-1]
        
        d = gui.dialogs.EditConstraint.Constraint(self, name, self.editingRegion.constraints[name][0], self.editingRegion.constraints[name][1])
        
        if d.result == None:
            return
        
        
        DataBoard.regions.removeRegion(self.editingRegion)
        self.editingRegion.setConstraint({name: d.result})
        DataBoard.regions.addRegion(self.editingRegion)
        
        gui.Utils.modified(atomic=True)
        
        self.refreshRegionConstraints()
        gui.Utils.redrawAllDesigners()
    
    def updateRegionName(self, event):
        if self.editingRegion == None:
            return
        val = self.regionNameEditor.get()
        if self.editingRegion.name == val:
            return
        
        self.editingRegion.name = val
        
        gui.Utils.modified()
        
        self.refreshRegionList()
        gui.Utils.updateDesignerStatusBars()
        
        for editor in DataBoard.boolEditors:
            editor.refreshRegionList()
        
        '''
        for designer in DataBoard.regEditors:
            designer.refreshRegionList()
            if not designer is self and designer.editingRegion is self.editingRegion:
                designer.refreshRegionName()
        '''
    def updateRegionPriority(self, event):
        if self.editingRegion == None:
            return
        try:
            val = int(self.regionPriorityEditor.get())
        except:
            return
        
        if self.editingRegion.priority == val:
            return
        self.editingRegion.priority = val
        gui.Utils.modified()
        
        '''
        for designer in DataBoard.regEditors:
            if not designer is self and designer.editingRegion is self.editingRegion:
                designer.refreshRegionPriority()
        '''
             
    def updateRegionSelection(self, event):
        if self.editingRegion == None:
            return
        val = self.regionSelectionOptionText.index(self.regionSelectionVar.get())+1
        if self.editingRegion.selection == val:
            return
        
        self.editingRegion.selection = val
        gui.Utils.modified(atomic=True)
        
        '''
        for designer in DataBoard.regEditors:
            if not designer is self and designer.editingRegion is self.editingRegion:
                designer.refreshRegionSelection()
        '''
        
    def updateRegionBounded(self, event):
        if self.editingRegion == None:
            return
        if self.regionBoundedOptionText.index(self.regionBoundedVar.get()) == 0:
            val = True
        else:
            val = False
        if self.editingRegion.bounded == val:
            return
        self.editingRegion.bounded = val
        gui.Utils.modified(atomic=True)
        '''
        for designer in DataBoard.regEditors:
            if not designer is self and designer.editingRegion is self.editingRegion:
                designer.refreshRegionBounded()
        '''
    def updateRegionRepeat(self, event):
        if self.editingRegion == None:
            return
        val = self.regionRepeat.get()
        if self.editingRegion.repeat == val:
            return
        self.editingRegion.repeat = val 
        gui.Utils.modified()
        '''
        for designer in DataBoard.regEditors:
            if not designer is self and designer.editingRegion is self.editingRegion:
                designer.refreshRegionRepeat()
        '''
        
    def updateRegionDelay(self, event):
        if self.editingRegion == None:
            return
        try:
            val = int(self.regionDelay.get())
        except:
            return
        if self.editingRegion.delay == val:
            return
        self.editingRegion.delay = val
        gui.Utils.modified()
        '''
        for designer in DataBoard.regEditors:
            if not designer is self and designer.editingRegion is self.editingRegion:
                designer.refreshRegionDelay()
        '''
        
    def updateRegionCommand(self, event):
        if self.editingRegion == None:
            return
        if DataBoard.extEditorManager.files.has_key(self.editingRegion.id):
            return
        val = self.regionCommandEditor.get(1.0, END)
        if self.editingRegion.commands.strip() == val.strip():
            return
        self.editingRegion.commands = val
        gui.Utils.modified()
        '''
        for designer in DataBoard.regEditors:
            if not designer is self and designer.editingRegion is self.editingRegion:
                designer.refreshRegionCommand()
        '''
    
    def launchExtEditor(self, event=None):
        if self.editingRegion == None:
            return
        if DataBoard.extEditorManager.files.has_key(self.editingRegion.id):
            DataBoard.extEditorManager.removeFile(self.editingRegion)
            #del self.editingRegion.temp['extEdit']
            self.refreshRegionCommand()
            gui.Utils.modified(atomic=True)
        else:
            result = DataBoard.extEditorManager.addFile(self.editingRegion, self.extEditorEntry.get(), self.editingRegion.name)
            if result == '1':
                #self.editingRegion.temp['extEdit'] = True
                self.refreshRegionCommand()
            else:
                tkMessageBox.showerror("Error", result)
         
        
        '''
        editor = self.extEditorEntry.get()
        
        
        if not self.editingRegion.temp.has_key('extEditFile'):
            fileName = self.editingRegion.id + str(uuid.uuid4())
            self.editingRegion.temp['extEditFile'] = fileName
        else:
            fileName = self.editingRegion.temp['extEditFile']
        
        filePath = os.path.join(tmpDir, fileName)
        
        F = open(filePath, 'w')
        F.write(self.editingRegion.commands)
        F.close()
        
        subprocess.Popen([editor, filePath])
        '''
        
    def deleteRegion(self, event=None):
        reg = self.editingRegion
        if reg.type == 'basic':
            DataBoard.regions.removeRegion(reg)
            #DataBoard.regions.dumpRegions()
        elif reg.type == 'boolean':
            del DataBoard.booleanRegions[reg.id]
            #print DataBoard.booleanRegions
        if reg.id == DataBoard.selectedRegion:
            #print reg.id,
            #print DataBoard.selectedRegion
            DataBoard.selectedRegion = ''
        
        gui.Utils.modified(atomic=True)
        #gui.Utils.updateRegionEditors()
        self.editingRegion = None
        gui.Utils.updateRegionEditors()
        '''
        for designer in DataBoard.regEditors:
            designer.updateRegionList()
            designer.updateSelectedRegion()
        '''
        gui.Utils.reinitAllDesigners()   
    
    def refreshRegionList(self):
        self.regionListBoxData = []
        self.editingRegion = None
        
        for index in DataBoard.regions.regions:
            for reg in DataBoard.regions.regions[index]:
                self.regionListBoxData.append(reg)
                
        for index in DataBoard.booleanRegions:
            reg = DataBoard.booleanRegions[index]
            self.regionListBoxData.append(reg)
            
        # Sort by region name
        self.regionListBoxData = sorted(self.regionListBoxData, key=lambda reg: reg.name)
        
        self.regionListBox.delete(1, END)
        selectedIndex = 0
        for reg in self.regionListBoxData:
            self.regionListBox.insert(END, reg.name)
            if reg.id == DataBoard.selectedRegion:
                selectedIndex = self.regionListBox.size()-1
                self.editingRegion = reg
        
        if selectedIndex > 0: 
            self.regionListBox.selection_set(first=selectedIndex)
        
    def updateSelectedRegion(self):
        if self.editingRegion == None:
            self.regionNameEditor.delete(0, END)
            self.regionNameEditor.config(state=DISABLED)
            self.regionPriorityEditor.delete(0, END)
            self.regionPriorityEditor.config(state=DISABLED)
            self.regionCommandEditor.delete(1.0, END)
            self.regionCommandEditor.config(state=DISABLED)
            self.regionSelectionOption.config(state=DISABLED)
            self.regionBoundedOption.config(state=DISABLED)
            
            # -- REPEAT AND DELAY
            self.regionRepeat.delete(0, END)
            self.regionRepeat.config(state=DISABLED)
            self.regionDelay.delete(0, END)
            self.regionDelay.config(state=DISABLED)
            
            self.regionConstraintDisplay.delete(1, END)
            self.deleteRegionButton.configure(state=DISABLED)
            
            self.extEditorEntry.configure(state=DISABLED)
            self.launchExtEditorButton.configure(state=DISABLED, text="Attach Editor")
            return
        
        self.regionNameEditor.config(state=NORMAL)
        self.regionPriorityEditor.config(state=NORMAL)
        self.regionSelectionOption.config(state=NORMAL)
        self.regionBoundedOption.config(state=NORMAL)
        self.regionRepeat.config(state=NORMAL)
        self.regionDelay.config(state=NORMAL)
        self.deleteRegionButton.configure(state=NORMAL)
        
        self.refreshRegionName()
        self.refreshRegionPriority()
        self.refreshRegionSelection()
        self.refreshRegionBounded()
        self.refreshRegionRepeat()
        self.refreshRegionDelay()
        self.refreshRegionCommand()
        self.refreshRegionConstraints()
        
    def refreshRegionName(self):
        self.regionNameEditor.delete(0, END)
        self.regionNameEditor.insert(0, self.editingRegion.name)
    
    def refreshRegionPriority(self):
        self.regionPriorityEditor.delete(0, END)
        self.regionPriorityEditor.insert(0, self.editingRegion.priority)
    
    def refreshRegionSelection(self):
        self.regionSelectionVar.set(self.regionSelectionOptionText[self.editingRegion.selection-1])
    
    def refreshRegionBounded(self):
        if self.editingRegion.bounded:
            self.regionBoundedVar.set(self.regionBoundedOptionText[0])
        else:
            self.regionBoundedVar.set(self.regionBoundedOptionText[1])
    
    def refreshRegionRepeat(self):
        self.regionRepeat.delete(0, END)
        self.regionRepeat.insert(0, self.editingRegion.repeat)
    
    def refreshRegionDelay(self):
        self.regionDelay.delete(0, END)
        self.regionDelay.insert(0, self.editingRegion.delay)
        
    def refreshRegionCommand(self):
        if DataBoard.extEditorManager.files.has_key(self.editingRegion.id):
        #if self.editingRegion.temp.has_key('extEdit'):
            self.regionCommandEditor.configure(state=NORMAL)
            self.regionCommandEditor.delete(1.0, END)
            self.regionCommandEditor.insert(END, self.editingRegion.commands)
            self.regionCommandEditor.config(state=DISABLED)
            self.extEditorEntry.configure(state=DISABLED)
            self.launchExtEditorButton.configure(state=NORMAL, text="Release Editor")
        else:
            self.regionCommandEditor.config(state=NORMAL)
            self.regionCommandEditor.delete(1.0, END)
            self.regionCommandEditor.insert(END, self.editingRegion.commands)
            self.extEditorEntry.configure(state=NORMAL)
            self.launchExtEditorButton.configure(state=NORMAL, text="Attach Editor")
        
        self.regionCommandEditor.delete(1.0, END)
        self.regionCommandEditor.insert(END, self.editingRegion.commands)
        
    def refreshRegionConstraints(self):
        self.regionConstraintDisplay.delete(1, END)
        if self.editingRegion.type == 'basic':
            self.constraintNames = []
            for cons in self.editingRegion.constraints:
                self.constraintNames.append(cons)
            self.constraintNames = sorted(self.constraintNames)
            for cons in self.constraintNames:
                lowerbound = self.editingRegion.constraints[cons][0] 
                upperbound = self.editingRegion.constraints[cons][1]
                if lowerbound == 'inf':
                    lowerbound = '-inf'
                else:
                    lowerbound = "%.2f" % lowerbound
                if upperbound == 'inf':
                    upperbound = '+inf'
                else:
                    upperbound = "%.2f" % upperbound
                self.regionConstraintDisplay.insert(END, cons + ' (' + Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars'][cons]['unit'] + ')' + ": %s, %s" %(lowerbound, upperbound))
        elif self.editingRegion.type == 'boolean':
            for item in self.editingRegion.union:
                self.regionConstraintDisplay.insert(END, 'union: ' + item.name)
            for item in self.editingRegion.exclusion:
                self.regionConstraintDisplay.insert(END, 'exclu: ' + item.name)
            for item in self.editingRegion.intersection:
                self.regionConstraintDisplay.insert(END, 'inter: ' + item.name)

    def closeWindow(self):
        gui.Utils.removeRegEditor(self)
        self.destroy()
        