'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on Dec 22, 2010

@author: Jimmy Li
'''

import Prototype
from Tkinter import *
import tkMessageBox
from config.Robots import Robots
from control.DataBoard import DataBoard
from control.Region import Region
from control.Waypoint import *
import gui.dialogs
import operator

class Designer2D(Prototype.DesignerTK):
    def __init__(self, master, settings): 
        self.dimensions = 2
        self.leftMarginWidth = 50 # Width of the space displaying the axis
        self.drawPadWidth = 600
        self.drawPadHeight = 600
        self.rightMarginWidth = 50
        self.bottomMarginHeight = 30
        self.canvasWidth = 700
        self.canvasHeight = 630
        self.drawingRegionIndex = 0
        self.verticalAxisOffset = 450
        self.horizontalAxisOffset = 400
        
        Prototype.DesignerTK.__init__(self, master, settings)
        
        self.attributeFrame = Frame(self, width=100, height=self.canvasHeight-1, background="#d9d9d9")
        self.attributeFrame.pack(side=TOP, fill=X)
        
        self.editingXYFrame = Frame(self.attributeFrame)
        self.editingXYFrame.grid(row=0, column=0, sticky=W)
        #Label(self.editingXYFrame, text="Edit").grid(sticky=E, row=0, column=0, padx=2, pady=2)
        
        self.editingObjectRadioFrame = Frame(self.editingXYFrame)
        self.editingObjectRadioFrame.grid(row=0, column=0, sticky=W, padx=15, pady=3)
        
        self.editingObjectRegion = Radiobutton(self.editingObjectRadioFrame, text="Region", variable=self.editingObject, value=1)
        self.editingObjectRegion.pack(side=LEFT)
        self.editingObjectWaypoint = Radiobutton(self.editingObjectRadioFrame, text="Waypoint", variable=self.editingObject, value=2)
        self.editingObjectWaypoint.pack(side=LEFT)
       
        Label(self.editingXYFrame, text="Edit(X):").grid(row=0, column=1, padx=2, pady=2)
        Label(self.editingXYFrame, text="Edit(Y):").grid(row=0, column=3, padx=2, pady=2)
        
        self.editingParamLabels['X'] = StringVar(self.editingXYFrame)
        self.editingParamLabels['Y'] = StringVar(self.editingXYFrame)
        
        canvasframe = Frame(self)
        self.canvasframe = canvasframe
        self.canvas = Canvas(canvasframe, highlightthickness=0, width=self.canvasWidth, height=(self.canvasHeight-1), bg='white')
        
        self.canvas.create_rectangle(0,0,self.leftMarginWidth,self.drawPadHeight, fill="#F0F0F0", outline = "#F0F0F0", tags=("margin", "leftMargin"))
        
        self.canvas.create_rectangle(self.leftMarginWidth+1,0,self.leftMarginWidth+self.drawPadWidth,self.drawPadHeight, fill="#FFFFFF", outline = "#FFFFFF", tags=("drawPad"))
        
        self.canvas.create_rectangle(self.leftMarginWidth + self.drawPadWidth,0,self.leftMarginWidth+self.drawPadWidth+self.rightMarginWidth, self.canvasHeight, fill="#F0F0F0", outline = "#F0F0F0", tags=("margin", "rightMargin"))
        
        self.canvas.create_rectangle(0,self.drawPadHeight+1,self.leftMarginWidth+self.drawPadWidth,self.canvasHeight, fill="#F0F0F0", outline = "#F0F0F0", tags=("margin", "bottomMargin"))
        
        self.canvas.pack()
        canvasframe.pack(side=TOP)
        
        self.createStatusBar()
        
        self.registerListeners()
        
        self.drawAxis()
        
        self.processSettings()
    
    def updateEditingParam(self):
        if self.editingParamCreated:
            self.editingParamMenuX.grid_forget()
            self.editingParamMenuY.grid_forget()
        
        self.editingParamLabels['X'].set(self.editingParameters['X'] + ' (' + Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars'][self.editingParameters['X']]['unit'] + ')') # default value
        self.editingParamLabels['Y'].set(self.editingParameters['Y'] + ' (' + Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars'][self.editingParameters['Y']]['unit'] + ')') # default value
        
        self.editingParamMenuX = OptionMenu(self.editingXYFrame, self.editingParamLabels['X'], *tuple(self.visualizingParamLabels['X']), command=self.changeEditingParam)
        self.editingParamMenuX.grid(row=0, column=2, pady=2)
        self.editingParamMenuY = OptionMenu(self.editingXYFrame, self.editingParamLabels['Y'], *tuple(self.visualizingParamLabels['Y']), command=self.changeEditingParam)
        self.editingParamMenuY.grid(row=0, column=4, pady=2)
        self.editingParamCreated = True
    
    def changeEditingParam(self, event):
        ind = self.visualizingParamLabels['X'].index(self.editingParamLabels['X'].get())
        self.editingParameters['X'] = self.visualizingParameters['X'][ind] 
        ind = self.visualizingParamLabels['Y'].index(self.editingParamLabels['Y'].get())
        self.editingParameters['Y'] = self.visualizingParameters['Y'][ind]
        
    def settingsDialog(self):
        d = gui.dialogs.DesignerSettings.DesignerSettings2D(self)
        self.settings = d.result
        if self.settings == None:
            return
        self.processSettings()
        
    def initVisualization(self):
        self.visualizingRegions = {}
        
        for param in self.visualizingParameters['X']: 
            for reg in DataBoard.regions.getReleventRegions([param]):
                self.visualizingRegions[reg.id] = reg
                
        for param in self.visualizingParameters['Y']: 
            for reg in DataBoard.regions.getReleventRegions([param]):
                self.visualizingRegions[reg.id] = reg
        
        self.removeRobotImage()
                         
    def redrawVisualization(self):
        self.canvas.delete("region")
                
        orderedBySize = []
        
        for regID in self.visualizingRegions:
            reg = self.visualizingRegions[regID]
            reg.temp['visualized'] = False

        # Process regions with constraints in both axes    
        for regID in self.visualizingRegions:
            reg = self.visualizingRegions[regID]
            for constr in reg.constraints:
                for constr2 in reg.constraints:
                    inX = constr in self.visualizingParameters['X']
                    inY = constr2 in self.visualizingParameters['Y']
                    if inX and inY:      
                        upperboundX = reg.constraints[constr][1]
                        lowerboundX = reg.constraints[constr][0]
                        upperboundY = reg.constraints[constr2][1]
                        lowerboundY = reg.constraints[constr2][0]                        
                        surfaceArea = self.drawExistingRegion(reg.id, reg.color, upperboundX, lowerboundX, upperboundY, lowerboundY)
                        orderedBySize.append((surfaceArea, reg.id))
                        reg.temp['visualized'] = True 
        
        # Process regions with constraints in only one of the axes
        for regID in self.visualizingRegions:
            reg = self.visualizingRegions[regID]
            if not reg.temp['visualized']: 
                for constr in reg.constraints:
                    inX = constr in self.visualizingParameters['X']
                    inY = constr in self.visualizingParameters['Y']
                    if inX or inY:
                        if inX:
                            upperboundX = reg.constraints[constr][1]
                            lowerboundX = reg.constraints[constr][0]
                            upperboundY = 'inf'
                            lowerboundY = 'inf'
                        elif inY:
                            upperboundY = reg.constraints[constr][1]
                            lowerboundY = reg.constraints[constr][0]
                            upperboundX = 'inf'
                            lowerboundX = 'inf'
                        
                        surfaceArea = self.drawExistingRegion(reg.id, reg.color, upperboundX, lowerboundX, upperboundY, lowerboundY)
                        orderedBySize.append((surfaceArea, reg.id))
                        reg.temp['visualized'] = True
                    
        # Now, display the regions in an order such that larger regions are below smaller regions
        orderedBySize = sorted(orderedBySize, key=operator.itemgetter(0), reverse=False)

        for entry in orderedBySize:
            self.canvas.tag_raise(entry[1])
            
        self.canvas.delete("waypoint")
        self.canvas.delete("waypointLine")
        
        for setID in DataBoard.waypointSets:
            waypointSet = DataBoard.waypointSets[setID]
            lastPositionH = 0
            lastPositionV = 0
            lastDrawnWaypointIndex = -1
            for paramX in self.visualizingParameters['X']:
                for paramY in self.visualizingParameters['Y']:
                    for i in range(0, len(waypointSet.waypoints)):
                        if waypointSet.waypoints[i].position.has_key(paramX) and waypointSet.waypoints[i].position.has_key(paramY):
                            #print 'drawn', i
                            positionH = self.unitToPixelX(waypointSet.waypoints[i].position[paramX])
                            positionV = self.unitToPixelY(waypointSet.waypoints[i].position[paramY])
                            if lastDrawnWaypointIndex >= 0:
                                if lastDrawnWaypointIndex == (i - 1):
                                    self.canvas.create_line(positionH, positionV, lastPositionH, lastPositionV, fill=waypointSet.color, width=2.0, smooth=1, tags=("waypointLine"))
                                else:
                                    self.canvas.create_line(positionH, positionV, lastPositionH, lastPositionV, fill=waypointSet.color, dash=(3, 5), width=2.0, tags=("waypointLine"))

                            self.canvas.create_oval(positionH-10, positionV-10, positionH+10, positionV+10, fill=waypointSet.color, tags=("waypoint", waypointSet.id, str(i+1), "waypointCircle", "WC"+waypointSet.id+str(i+1)))
                            self.canvas.create_text(positionH, positionV-7, anchor=N, text=str(i+1), tags=("waypoint", waypointSet.id, str(i+1)))
                            self.canvas.tag_raise("waypoint")
                            lastPositionH = positionH
                            lastPositionV = positionV
                            lastDrawnWaypointIndex = i
        
        self.markSelectionInDesigners()
        
        self.canvas.tag_raise("axis")
        self.canvas.tag_raise("margin")
        
        self.updateStatusBar()
    
    
    def redrawRobotPosition(self, position):
        if not self.drawingRobot:
            self.canvas.delete('robotImage')
            
        #TODO: There are a lot of redundancies in the following two 
        # for loops. Resolve this issue at some point
        #print '-------------'
        for paramx in self.visualizingParameters['X']:
            indexx = self.visualizingParameters['X'].index(paramx)
            for paramy in self.visualizingParameters['Y']:
                #print 'looping for', paramx, paramy
                if position.has_key(paramx) and position.has_key(paramy):
                    #print '  has_both'
                    indexy = self.visualizingParameters['Y'].index(paramy)
                    valx = position[paramx]
                    valy = position[paramy]
                    pixx = self.unitToPixelX(valx)
                    pixy = self.unitToPixelY(valy)
                    #print '  self.drawingRobot: ', self.drawingRobot
                    if not self.drawingRobot:
                        #print '  creating robot image for', paramx, paramy
                        #self.canvas.create_image(pixx, pixy, image=self.robotImage, tags=("robotImage", 'robotImage'+str(indexx)+str(indexy)))
                        self.canvas.create_oval(pixx - self.robotImageRadius, pixy-self.robotImageRadius, pixx + self.robotImageRadius, pixy + self.robotImageRadius, fill = self.robotImageColor, outline=self.robotImageColor, tags=("robotImage", 'robotImage'+str(indexx)+str(indexy)))
                    else:
                        #print '  moving robot image for', paramx, paramy
                        self.canvas.move('robotImage'+str(indexx)+str(indexy), pixx - self.robotImagePos['X'][paramx], pixy - self.robotImagePos['Y'][paramy])
        
        # Update self.robotImagePos
        for paramx in self.visualizingParameters['X']:
            indexx = self.visualizingParameters['X'].index(paramx)
            for paramy in self.visualizingParameters['Y']:
                if position.has_key(paramx) and position.has_key(paramy):
                    indexy = self.visualizingParameters['Y'].index(paramy)
                    valx = position[paramx]
                    valy = position[paramy]
                    pixx = self.unitToPixelX(valx)
                    pixy = self.unitToPixelY(valy)
                    self.robotImagePos['Y'][paramy] = pixy
                    self.robotImagePos['X'][paramx] = pixx
            
        self.canvas.tag_raise('robotImage')
            
        self.drawingRobot = True
    
    
    def drawExistingRegion(self, regid, color, upperboundX, lowerboundX, upperboundY, lowerboundY):
        '''
        print "drawing: ",
        print upperboundX,
        print lowerboundX,
        print upperboundY,
        print lowerboundY
        '''
        #Horizontal Axis
        if upperboundX == 'inf':
            right = self.drawPadWidth + self.leftMarginWidth + 1
        else:
            right = self.unitToPixelX(upperboundX)
        
        if lowerboundX == 'inf':
            left = -1
        else:
            left = self.unitToPixelX(lowerboundX)
    
    
        #Vertical Axis
        if upperboundY == 'inf':
            top = -1
        else:
            top = self.unitToPixelY(upperboundY)
        
        if lowerboundY == 'inf':
            bottom = self.drawPadHeight + 2
        else:
            bottom = self.unitToPixelY(lowerboundY)
        
        self.canvas.create_rectangle(left, bottom, right, top, fill=color, outline=self.regionOutline, stipple="gray25", tags=("region", regid))
        return (right - left)*(top - bottom) # surface area
        
    def drawCursor(self, event):
        if self.showCursor:
            y = self.canvas.canvasy(event.y)
            x = self.canvas.canvasy(event.x)
            if x <= self.leftMarginWidth or x > self.leftMarginWidth + self.drawPadWidth:
                return
            if y > self.drawPadHeight:
                return
            self.canvas.delete("cursor")
            self.canvas.create_line(self.leftMarginWidth+1, y, self.leftMarginWidth+self.drawPadWidth, y, fill="#FF3333", tags=("cursor"))
            measurementY = (self.zeroY-y)/self.pixelPerUnit;
            self.canvas.create_line(x, 0, x, self.drawPadHeight, fill="#FF3333", tags=("cursor"))
            measurementX = (x-self.zeroX-self.leftMarginWidth)/self.pixelPerUnit;
            self.canvas.create_text(self.leftMarginWidth+self.drawPadWidth+3, y, text="%.2f" % measurementY, fill="#FF3333", anchor=W, tags=("cursor"))
            self.canvas.create_text(x, self.drawPadHeight + 3, text="%.2f" % measurementX, fill="#FF3333", anchor=N, tags=("cursor"))
    
    def setZoomIndex(self, event):
        Prototype.DesignerTK.setZoomIndex(self, event)
        self.centerPixelX = self.canvas.canvasx(event.x)
        self.centerPixelY = self.canvas.canvasx(event.y)
        self.centerUnitX = self.pixelToUnitX(self.centerPixelX)
        self.centerUnitY = self.pixelToUnitY(self.centerPixelY)

    def zoomRedraw(self):
        if self.pixelPerUnit != self.pixelPerUnitPrev:
            self.zeroX = self.centerPixelX - self.leftMarginWidth - self.centerUnitX*self.pixelPerUnit # Derived from pixelToUnitX()
            self.zeroY = self.centerUnitY*self.pixelPerUnit + self.centerPixelY # Derived from pixelToUnitY()
        Prototype.DesignerTK.zoomRedraw(self)

    def initDrawRegion(self, event):
        self.drawingRegionIndexY = self.canvas.canvasy(event.y)
        self.drawingRegionIndexX = self.canvas.canvasx(event.x)
        self.drawingRegion = True
        
    def drawRegion(self, event):
        self.drawCursor(event)
        canvasy = self.canvas.canvasy(event.y)
        canvasx = self.canvas.canvasx(event.x)
        lengthY = canvasy - self.drawingRegionIndexY
        lengthX = canvasx - self.drawingRegionIndexX
        self.canvas.delete("drawingRegion")
        if lengthY >= 0:
            rectBoundTop = self.drawingRegionIndexY
            rectBoundBottom = canvasy
        else:
            rectBoundTop = canvasy
            rectBoundBottom = self.drawingRegionIndexY
        
        if lengthX >= 0:
            rectBoundLeft = self.drawingRegionIndexX
            rectBoundRight = canvasx
        else:
            rectBoundLeft = self.drawingRegionIndexX
            rectBoundRight = canvasx
        
        self.canvas.create_rectangle(rectBoundLeft, rectBoundTop, rectBoundRight, rectBoundBottom, fill="#FFFF66", outline="#FFFF66", tags=("drawingRegion", "cursorActive"))
    
    def endAddRegionDim(self, event):
        '''
        Add dimension(s) to a region
        '''
        if self.drawingRegion:
            self.drawingRegion = False
            self.canvas.delete("drawingRegion")
            
            # Do not add dimensions to the selected region if the region already has constraints
            # in the current designer
            if self.editingParameters['X'] == self.editingParameters['Y']:
                print "Error: Editing the same parameter in X and Y."
                return
            region = DataBoard.regions.getRegionByID(DataBoard.selectedRegion)
            if region == None:
                return
        
            canvasy = self.canvas.canvasy(event.y)
            canvasx = self.canvas.canvasx(event.x)
            lengthY = canvasy - self.drawingRegionIndexY
            lengthX = canvasx - self.drawingRegionIndexX
            self.canvas.delete("drawingRegion")
                        
            if not canvasy == self.drawingRegionIndexY and not canvasx == self.drawingRegionIndexX:

                if lengthY >= 0:
                    rectBoundTop = self.drawingRegionIndexY
                    rectBoundBottom = canvasy
                else:
                    rectBoundTop = canvasy
                    rectBoundBottom = self.drawingRegionIndexY
                
                if lengthX >= 0:
                    rectBoundLeft = self.drawingRegionIndexX
                    rectBoundRight = canvasx
                else:
                    rectBoundLeft = self.drawingRegionIndexX
                    rectBoundRight = canvasx
                
                DataBoard.regions.removeRegion(region)
                
                region.setConstraint({self.editingParameters['X'] : [self.pixelToUnitX(rectBoundLeft), self.pixelToUnitX(rectBoundRight)],
                                         self.editingParameters['Y'] : [self.pixelToUnitY(rectBoundTop), self.pixelToUnitY(rectBoundBottom)]})

                DataBoard.regions.addRegion(region)
                self.updateSelectedRegion(region.id)
                gui.Utils.reinitAllDesigners()
                gui.Utils.modified(atomic=True)
                
                self.canvas.tag_raise("axis")
                self.canvas.tag_raise("margin")
                    
    def endDrawRegion(self, event):
        if self.drawingRegion:
            self.drawingRegion = False
            self.canvas.delete("drawingRegion")
            
            if self.editingParameters['X'] == self.editingParameters['Y']:
                print "Error: Editing the same parameter in X and Y."
                return
            
            canvasy = self.canvas.canvasy(event.y)
            canvasx = self.canvas.canvasx(event.x)
            lengthY = canvasy - self.drawingRegionIndexY
            lengthX = canvasx - self.drawingRegionIndexX         
            
            if not canvasy == self.drawingRegionIndexY and not canvasx == self.drawingRegionIndexX:
                
                
                if lengthY >= 0:
                    rectBoundTop = self.drawingRegionIndexY
                    rectBoundBottom = canvasy
                else:
                    rectBoundTop = canvasy
                    rectBoundBottom = self.drawingRegionIndexY
                
                if lengthX >= 0:
                    rectBoundLeft = self.drawingRegionIndexX
                    rectBoundRight = canvasx
                else:
                    rectBoundLeft = self.drawingRegionIndexX
                    rectBoundRight = canvasx
                
                newRegion = Region()
                
                newRegion.setConstraint({self.editingParameters['X'] : [self.pixelToUnitX(rectBoundLeft), self.pixelToUnitX(rectBoundRight)],
                                         self.editingParameters['Y'] : [self.pixelToUnitY(rectBoundTop), self.pixelToUnitY(rectBoundBottom)]})

                DataBoard.regions.addRegion(newRegion)
                self.updateSelectedRegion(newRegion.id)
                gui.Utils.reinitAllDesigners()
                gui.Utils.modified(atomic=True)
                self.canvas.tag_raise("axis")
                self.canvas.tag_raise("margin")
            
    
    def deleteRegionDim(self, event):
        item = event.widget.find_closest(event.x+1, event.y+1)
        itemID = self.canvas.gettags(item)[1]        
        reg = DataBoard.regions.getRegionByID(itemID)
        DataBoard.regions.removeRegion(reg)
        reg.setConstraint({self.editingParameters['X']: None, self.editingParameters['Y']: None})
        DataBoard.regions.addRegion(reg)
        gui.Utils.updateRegionEditors()
        gui.Utils.reinitAllDesigners()
        gui.Utils.modified(atomic=True)
    
    def deleteWaypointDim(self, event):
        item = event.widget.find_closest(event.x+1, event.y+1)
        itemID = self.canvas.gettags(item)[1]
        itemNo = self.canvas.gettags(item)[2]
        waypoints = DataBoard.waypointSets[itemID].waypoints
        waypoint = waypoints[int(itemNo)-1]
        del waypoint.position[self.editingParameters['X']]
        del waypoint.position[self.editingParameters['Y']]
        
        self.updateSelectedWaypointSet()
        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
        
    def moveWaypoint(self, event):
        self.drawCursor(event)
        canvasy = self.canvas.canvasy(event.y)
        canvasx = self.canvas.canvasy(event.x)
        self.canvas.delete("movingWaypoint")
        self.canvas.create_oval(canvasx-10, canvasy-10, canvasx+10, canvasy+10, fill="#FFFF66", outline="#FFFF66", tags=("movingWaypoint"))
    
    def endMovingWaypoint(self, event):
        self.movingWaypoint = False
        canvasy = self.canvas.canvasy(event.y)
        canvasx = self.canvas.canvasy(event.x)
        if self.moveWaypointIndexY == canvasy and self.moveWaypointIndexX == canvasx:
            return
        self.canvas.delete("movingWaypoint")
        if not self.checkEditingParamWritable():
            return;
        
        
        waypointSetID = self.canvas.gettags(self.waypointBeingMoved)[1]
        waypointNo = self.canvas.gettags(self.waypointBeingMoved)[2]
        waypoint = DataBoard.waypointSets[waypointSetID].waypoints[int(waypointNo)-1]
        waypoint.position[self.editingParameters['X']] = self.pixelToUnitX(canvasx)
        waypoint.position[self.editingParameters['Y']] = self.pixelToUnitY(canvasy)

        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
        
    def createNewWaypointSet(self, event):
        if not self.checkEditingParamWritable():
            return
        
        
        canvasy = self.canvas.canvasy(event.y)
        canvasx = self.canvas.canvasy(event.x)
        newWaypointSet = WaypointSet()
        waypoint = Waypoint()
        waypoint.position[self.editingParameters['Y']] = self.pixelToUnitY(canvasy)
        waypoint.position[self.editingParameters['X']] = self.pixelToUnitX(canvasx)
        newWaypointSet.waypoints.append(waypoint)
        DataBoard.waypointSets[newWaypointSet.id] = newWaypointSet
        DataBoard.selectedWaypointSet = newWaypointSet.id
        DataBoard.selectedWaypoint = 1
        
        self.updateSelectedWaypointSet()
        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
    
    def addToWaypointSet(self, event):
        if not self.checkEditingParamWritable():
            return
        canvasy = self.canvas.canvasy(event.y)
        canvasx = self.canvas.canvasy(event.x)
        if not DataBoard.waypointSets.has_key(DataBoard.selectedWaypointSet):
            return
        
        
        waypointSet = DataBoard.waypointSets[DataBoard.selectedWaypointSet]
        waypoint = Waypoint()
        waypoint.position[self.editingParameters['Y']] = self.pixelToUnitY(canvasy)
        waypoint.position[self.editingParameters['X']] = self.pixelToUnitX(canvasx)
        index = int(DataBoard.selectedWaypoint)
        waypointSet.waypoints.insert(index, waypoint)
        DataBoard.selectedWaypoint = index + 1
        self.updateSelectedWaypointSet()
        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
        
    
    def addToWaypointDim(self, event):
        if not self.checkEditingParamWritable():
            return
        canvasy = self.canvas.canvasy(event.y)
        canvasx = self.canvas.canvasy(event.x)
        if not DataBoard.waypointSets.has_key(DataBoard.selectedWaypointSet):
            return
        
        
        waypointSet = DataBoard.waypointSets[DataBoard.selectedWaypointSet]
        waypointIndex = int(DataBoard.selectedWaypoint) - 1
        waypoint = waypointSet.waypoints[waypointIndex]
        waypoint.position[self.editingParameters['Y']] = self.pixelToUnitY(canvasy)
        waypoint.position[self.editingParameters['X']] = self.pixelToUnitX(canvasx)
        
        gui.Utils.updateWaypointEditors()
        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
        
    def updateIfVisualizing(self, parameters):
        reInited = False
        matchX = matchY = False
        for param in self.visualizingParameters['X']:
            if param in parameters:
                matchX = True
                break
        for param in self.visualizingParameters['Y']:
            if param in parameters:
                matchY = True
                break
        if matchX and matchY:
            self.initVisualization()
            self.redrawVisualization()
            reInited = True

        if not reInited:
            self.redrawVisualization()

    
    def pixelToUnitX(self, pixel):
        return (pixel - (self.zeroX + self.leftMarginWidth)) / self.pixelPerUnit
    
    def unitToPixelX(self, unit):
        return (unit * self.pixelPerUnit) + (self.leftMarginWidth + self.zeroX) 
    
    def pixelToUnitY(self, pixel):
        return (self.zeroY - pixel) / self.pixelPerUnit
    
    def unitToPixelY(self, unit):
        return self.zeroY - (unit * self.pixelPerUnit)
    
    def drawAxis(self):
        self.drawVerticalAxis(self.zeroX + self.leftMarginWidth, drawZero = False)
        self.drawHorizontalAxis(self.zeroY, drawZero = False)
    
    def checkEditingParamWritable(self):
        stateVarsDic = Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars']
        if not stateVarsDic[self.editingParameters['X']]['writable']:
            tkMessageBox.showerror("Error", "You are editing " + self.editingParameters['X'] + " on the horizontal axis. This is not a writable state variable.")
            return False
        if not stateVarsDic[self.editingParameters['Y']]['writable']:
            tkMessageBox.showerror("Error", "You are editing " + self.editingParameters['Y'] + " on the vertical axis. This is not a writable state variable.")
            return False
        
        return True