'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on 2010-05-26

@author: Jimmy Li
'''

from Tkinter import *
import gui.dialogs.DesignerSettings
from config.Robots import Robots 
from control.DataBoard import DataBoard
import math
import gui.Utils
import control.Region
import control.Waypoint
import operator
import os
#import ImageTk

class DesignerTK(Toplevel):
    def __init__(self, master, settings):
        Toplevel.__init__(self, master)
        
        self.master = master
        self.settings = settings
        self.zooming = False
        self.movingScale = False
        self.showCursor = True
        self.drawingRegion = False
        self.minpixelPerUnit = 0.5
        self.maxpixelPerUnit = 18
        self.pixelPerUnit = 5
        self.zeroY = self.drawPadHeight / 2
        self.zeroX = self.drawPadWidth / 2
        self.visualizingParameters = {}
        self.visualizingParamLabels = {}
        self.editingObject = IntVar() # 1 = region, 2 = waypoint
        self.editingObject.set(1)
        self.editingParameters = {}
        self.editingParamLabels = {}
        self.editingParamCreated = False
        self.visualizingRegions = {}
        self.regionOutline = '#333300'
        self.selectedRegionOutline = '#FF2B2B'
        self.waypointOutline = '#333300'
        self.selectedWaypointOutline = '#FF2B2B'
        self.movingWaypoint = False
        self.waypointBeingMoved = None
        self.visualizeWaypoint = False
        self.moveWaypointIndexX = 0
        self.moveWaypointIndexY = 0
        self.unselectedOutlineWidth = 1
        self.selectedOutlineWidth = 2
        
        #self.robotImage = ImageTk.PhotoImage(file=os.path.join(gui.Utils.getImagePath(), 'robot.png'))
        self.drawingRobot = False
        self.robotImagePos = {'X':{}, 'Y':{}}
        self.robotImageRadius = 10
        self.robotImageColor = '#FF0000'
        
        #self.tk.call('wm', 'iconphoto', self._w, DataBoard.icon)

        self.protocol("WM_DELETE_WINDOW", self.closeWindow)
        
        self.resizable(width=FALSE, height=FALSE)
        
        self.geometry("+%d+%d" % (master.winfo_rootx()+50,
                                  master.winfo_rooty()+50))
        self.createMenu()     
    
    def closeWindow(self):
        gui.Utils.removeDesigner(self)
        self.destroy()
    
    def createMenu(self):
        menu = Menu(self)
        self.config(menu=menu)
        
        options = Menu(menu)
        menu.add_cascade(label='Options', menu=options)
        options.add_command(label='Settings', command=self.settingsDialog)
    
    def processSettings(self):
        title = []
        for axis in self.settings:
            self.visualizingParameters[axis] = []
            self.visualizingParamLabels[axis] = []
            
            '''
            #Sort
            keys = []
            for param in self.settings[axis]:
                keys.append(param)
            keys = sorted(keys)
            '''
            
            firstOfAxis = True
            for param in self.settings[axis]:
                self.visualizingParameters[axis].append(param)
                if firstOfAxis:
                    firstOfAxis = False
                    self.editingParameters[axis] = param #Choose the first parameter to be the one that is being edited
                self.visualizingParamLabels[axis].append(param + ' (' + Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars'][param]['unit'] + ')')
                title.append(param + ' (' + Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars'][param]['unit'] + ')')
        
        title = sorted(title)
        self.title(", ".join(title))
        self.initVisualization()
        self.updateEditingParam()
        self.redrawVisualization()
        
    def registerListeners(self):        
        
        self.bind("<Control-KeyRelease-z>", gui.Utils.undo)
        self.bind("<Control-KeyRelease-r>", gui.Utils.redo)
        
        self.canvas.bind("<Button-3>", self.setZoomIndex)
        self.canvas.bind("<ButtonRelease-3>",  self.endZoom) #redraw itinerary on zoom release as doing it before is too CPU expensive
        
        self.canvas.bind("<ButtonRelease-1>", self.leftButtonRelease)
        self.canvas.bind("<Motion>", self.mouseMotion)
        self.canvas.bind("<Button-1>", self.leftButton)
        self.canvas.bind("<Double-Button-1>", self.doubleLeftButton)
        
        self.canvas.bind("<Control-ButtonRelease-1>", self.controlMouseRelease)
        self.canvas.bind("<Control-Button-1>", self.controlMouseButton)
        self.canvas.bind("<Shift-Button-1>", self.shiftMouseButton)
        
        self.canvas.bind("<Button-2>", self.middleButton)
        
        self.canvas.bind("<Up>", self.upKey)
        self.canvas.bind("<Down>", self.downKey)
        self.canvas.bind("<Left>", self.leftKey)
        self.canvas.bind("<Right>", self.rightKey)
    
    def doubleLeftButton(self, event):
        #TODO: Read below
        '''
        This feature needs work. The idea is that by double clicking
        a region, the region is sent to the back of the stack, revealing
        the region(s) that were below it. However, this comes into 
        conflict with another feature, where smaller regions are automatically
        brought to the top. The existence of this conflict is preventing
        this feature from functioning. This needs to be resolved.
        '''
        #print 'dbclick'
        if not self.editingObject.get() == 1:
            return
        item = event.widget.find_closest(event.x+1, event.y+1)
        tags = self.canvas.gettags(item)
        tag = tags[0]
        if not tag == 'region':
            return
        regid = tags[1]
        self.canvas.tag_lower(regid)
        self.canvas.tag_lower('drawPad')
        #self.clickRegion(event)
    
    def upKey(self, event):
        self.expandToInf('up')
    
    def downKey(self, event):
        self.expandToInf('down')
        
    def leftKey(self, event):
        self.expandToInf('left')
        
    def rightKey(self, event):
        self.expandToInf('right')
    
    def expandToInf(self, direction):
        regid = DataBoard.selectedRegion
        if regid == '':
            return
        region = DataBoard.regions.getRegionByID(regid)
        DataBoard.regions.removeRegion(region)
        
        if self.dimensions == 1:
            if direction == 'up':
                lowerbound = region.constraints[self.editingParameters['X']][0]
                region.setConstraint({self.editingParameters['X']:[lowerbound, 'inf']})
            elif direction == 'down':
                upperbound = region.constraints[self.editingParameters['X']][1]
                region.setConstraint({self.editingParameters['X']:['inf', upperbound]})
        else:
            if direction == 'up':
                lowerbound = region.constraints[self.editingParameters['Y']][0]
                region.setConstraint({self.editingParameters['Y']:[lowerbound, 'inf']})
            elif direction == 'down':
                upperbound = region.constraints[self.editingParameters['Y']][1]
                region.setConstraint({self.editingParameters['Y']:['inf', upperbound]})
            elif direction == 'right':
                lowerbound = region.constraints[self.editingParameters['X']][0]
                region.setConstraint({self.editingParameters['X']:[lowerbound, 'inf']})
            elif direction == 'left':
                upperbound = region.constraints[self.editingParameters['X']][1]
                region.setConstraint({self.editingParameters['X']:['inf', upperbound]})
        
        DataBoard.regions.addRegion(region)
        
        gui.Utils.updateRegionEditors()
        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
        
           
    def setZoomIndex(self, event):
        self.zooming = True
        self.zoomIndex = event.y
    
    def endZoom(self, event):
        self.zooming = False
    
    def zoom(self, event):
        self.pixelPerUnitPrev = self.pixelPerUnit
        
        zoomBy = event.y - self.zoomIndex
        if math.fabs(zoomBy) < 4:
            return
        if zoomBy < 0:
            if self.pixelPerUnit >= 1:
                self.pixelPerUnit += 0.5
            else:
                self.pixelPerUnit += 0.05
            if self.pixelPerUnit > self.maxpixelPerUnit:
                self.pixelPerUnit = self.maxpixelPerUnit
        elif zoomBy > 0:
            if self.pixelPerUnit > 1:
                self.pixelPerUnit -= 0.5
            else:
                self.pixelPerUnit -= 0.05
            if self.pixelPerUnit < self.minpixelPerUnit:
                self.pixelPerUnit = self.minpixelPerUnit
        else:
            return

        self.zoomIndex = event.y
        self.zoomRedraw()

    def zoomRedraw(self):
        self.canvas.delete("axis")
        self.drawAxis()
        self.redrawVisualization()
            
    def initMoveScale(self, event):
        self.movingScale = True
        self.moveScaleIndexY = event.y
        self.moveScaleIndexX = event.x
    
    def mouseMotion(self, event):
        self.drawCursor(event)
        
        if self.zooming:
            self.zoom(event)
        
        if self.movingScale:
            self.moveScale(event)
            
        if self.drawingRegion:
            self.drawRegion(event)
        
        if self.movingWaypoint:
            self.moveWaypoint(event)
    
    def leftButton(self,event):
        self.canvas.focus_set()
        item = event.widget.find_closest(event.x+1, event.y+1)
        tags = self.canvas.gettags(item)
        tag = tags[0]
        if self.editingObject.get() == 1: # 1 = region, 2 = waypoint
            if tag == 'drawPad' or tag == 'region':
                self.initDrawRegion(event)
            if tag == 'region':
                self.clickRegion(event)
        elif self.editingObject.get() == 2: # 1 = region, 2 = waypoint
            if tag == 'drawPad' or tag == 'region':
                self.createNewWaypointSet(event)
            if tag == 'waypoint':
                self.clickWaypoint(event)
                #print 'tags', tags
        if tag == 'axis':
            self.initMoveScale(event)
    
    def middleButton(self, event):
        item = event.widget.find_closest(event.x+1, event.y+1)
        tag = self.canvas.gettags(item)[0]
        if tag == 'region':
            self.deleteRegionDim(event)
        if tag == 'waypoint':
            self.deleteWaypointDim(event)
    
    def controlMouseRelease(self, event):
        if self.editingObject.get() == 1: # 1 = region, 2 = waypoint
            self.endAddRegionDim(event)
    
    def controlMouseButton(self, event):
        item = event.widget.find_closest(event.x+1, event.y+1)
        tag = self.canvas.gettags(item)[0]
        if self.editingObject.get() == 1: # 1 = region, 2 = waypoint
            if tag == 'drawPad' or tag == 'region':
                self.initDrawRegion(event)
        if self.editingObject.get() == 2: # 1 = region, 2 = waypoint
            self.addToWaypointDim(event)
    
    def shiftMouseButton(self, event):
        if self.editingObject.get() == 2: # 1 = region, 2 = waypoint
            self.addToWaypointSet(event)
            
    def leftButtonRelease(self, event):
        if self.movingScale:
            self.movingScale = False
        
        if self.movingWaypoint:
            self.endMovingWaypoint(event)
        
        self.endDrawRegion(event)
                
    def leftButtonMove(self, event):
        if self.movingScale:
            self.moveScale(event)
        elif self.drawingRegion:
            self.drawRegion(event)
    
    def moveScale(self, event):
        moveByY = event.y - self.moveScaleIndexY
        moveByX = event.x - self.moveScaleIndexX
        self.zeroY += moveByY
        self.zeroX += moveByX
        self.moveScaleIndexY = event.y
        self.moveScaleIndexX = event.x
        self.canvas.delete("axis")
        self.drawAxis()
        self.redrawVisualization()
    
    def enableCursor(self, event):
        self.showCursor = True
        self.drawCursor(event)
        
    def disableCursor(self, event):
        self.showCursor = False
        self.canvas.delete("cursor")
        
    def drawVerticalAxis(self, xOffset, drawZero = True):
        scaleLineX = xOffset - 5
        self.canvas.create_line(xOffset,0,xOffset,self.drawPadHeight, fill="#0066CC", tags=("axis"))
        
        if drawZero:
            self.canvas.create_line(scaleLineX, self.zeroY, xOffset, self.zeroY, fill="#0066CC", tags=("axis"))
            self.canvas.create_text(scaleLineX-2, self.zeroY, anchor=E, text="0", fill="#0066CC", tags=("axis"))
          
        if self.pixelPerUnit < 1:
            self.scaleIncrement = 20
            self.scaleLabelIncrement = 100  
        elif self.pixelPerUnit < 3:
            self.scaleIncrement = 10
            self.scaleLabelIncrement = 40
        elif self.pixelPerUnit < 5:
            self.scaleIncrement = 5
            self.scaleLabelIncrement = 10
        elif self.pixelPerUnit < 12:
            self.scaleIncrement = 2
            self.scaleLabelIncrement = 10
        else:
            self.scaleIncrement = 1
            self.scaleLabelIncrement = 2

        for i in range(self.scaleIncrement, int((self.zeroY)/self.pixelPerUnit), self.scaleIncrement):
            yPos = self.zeroY - (i * self.pixelPerUnit)
            self.canvas.create_line(scaleLineX, yPos, xOffset, yPos, fill="#0066CC", tags=("axis"))
            
        for i in range(self.scaleIncrement, int((self.drawPadHeight-self.zeroY)/self.pixelPerUnit), self.scaleIncrement):
            yNeg = self.zeroY + (i * self.pixelPerUnit)
            self.canvas.create_line(scaleLineX, yNeg, xOffset, yNeg, fill="#0066CC", tags=("axis"))
        
        for i in range(self.scaleLabelIncrement, int((self.zeroY)/self.pixelPerUnit), self.scaleLabelIncrement):
            yPos = self.zeroY - (i * self.pixelPerUnit)
            self.canvas.create_text(scaleLineX-2, yPos, anchor=E, text="%d" % i, fill="#0066CC", tags=("axis"))
            
        for i in range(self.scaleLabelIncrement, int((self.drawPadHeight-self.zeroY)/self.pixelPerUnit), self.scaleLabelIncrement):
            yNeg = self.zeroY + (i * self.pixelPerUnit)
            self.canvas.create_text(scaleLineX-2, yNeg, anchor=E, text="-%d" % i, fill="#0066CC", tags=("axis"))
    
    def drawHorizontalAxis(self, yOffset, drawZero = True):
        scaleLineY = yOffset + 5
        self.canvas.create_line(self.leftMarginWidth, yOffset, self.drawPadWidth+self.leftMarginWidth, yOffset, fill="#0066CC", tags=("axis"))
        
        if drawZero:
            self.canvas.create_line(self.zeroX + self.leftMarginWidth, yOffset, self.zeroX + self.leftMarginWidth, scaleLineY, fill="#0066CC", tags=("axis"))
            self.canvas.create_text(self.zeroX + self.leftMarginWidth, scaleLineY + 2, anchor=N, text="0", fill="#0066CC", tags=("axis"))
        
        if self.pixelPerUnit < 1:
            self.scaleIncrement = 20
            self.scaleLabelIncrement = 100    
        elif self.pixelPerUnit < 3:
            self.scaleIncrement = 10
            self.scaleLabelIncrement = 40
        elif self.pixelPerUnit < 5:
            self.scaleIncrement = 5
            self.scaleLabelIncrement = 10
        elif self.pixelPerUnit < 12:
            self.scaleIncrement = 2
            self.scaleLabelIncrement = 10
        else:
            self.scaleIncrement = 1
            self.scaleLabelIncrement = 2

        for i in range(self.scaleIncrement, int((self.zeroX)/self.pixelPerUnit), self.scaleIncrement):
            xNeg = self.zeroX - (i * self.pixelPerUnit) + self.leftMarginWidth
            self.canvas.create_line(xNeg, yOffset, xNeg, scaleLineY, fill="#0066CC", tags=("axis"))
            
        for i in range(self.scaleIncrement, int((self.drawPadWidth-self.zeroX)/self.pixelPerUnit), self.scaleIncrement):
            xPos = self.zeroX + self.leftMarginWidth + (i * self.pixelPerUnit)
            self.canvas.create_line(xPos, yOffset, xPos, scaleLineY, fill="#0066CC", tags=("axis"))
        
        for i in range(self.scaleLabelIncrement, int((self.zeroX)/self.pixelPerUnit), self.scaleLabelIncrement):
            xNeg = self.zeroX - (i * self.pixelPerUnit) + self.leftMarginWidth
            self.canvas.create_text(xNeg, scaleLineY + 2, anchor=N, text="-%d" % i, fill="#0066CC", tags=("axis"))
                
        for i in range(self.scaleLabelIncrement, int((self.drawPadWidth-self.zeroX)/self.pixelPerUnit), self.scaleLabelIncrement):
            xPos = self.zeroX + (i * self.pixelPerUnit) + self.leftMarginWidth
            self.canvas.create_text(xPos, scaleLineY + 2, anchor=N, text="%d" % i, fill="#0066CC", tags=("axis"))
         
    def clickWaypoint(self, event):
        item = event.widget.find_closest(event.x+1, event.y+1)
        tags = self.canvas.gettags(item)
        id = tags[1]
        num = tags[2]
        DataBoard.selectedWaypointSet = id
        DataBoard.selectedWaypoint = int(num)
        
        self.updateSelectedWaypointSet()
        gui.Utils.redrawAllDesigners()
        
        self.movingWaypoint = True
        item = event.widget.find_closest(event.x+1, event.y+1)
        self.waypointBeingMoved = item
        self.moveWaypointIndexY = self.canvas.canvasy(event.y)
        self.moveWaypointIndexX = self.canvas.canvasy(event.x)
    
    def clickRegion(self, event):
        item = event.widget.find_closest(event.x+1, event.y+1)
        regid = self.canvas.gettags(item)[1]
        
        if not regid == DataBoard.selectedRegion:
            self.updateSelectedRegion(regid)
            gui.Utils.redrawAllDesigners()
        
        self.initDrawRegion(event)
    
    def deleteRegion(self, event):
        item = event.widget.find_closest(event.x+1, event.y+1)
        itemID = self.canvas.gettags(item)[1]        
        reg = DataBoard.regions.getRegionByID(itemID)
        DataBoard.regions.removeRegion(reg)
        DataBoard.selectedRegion = ''
        gui.Utils.reinitAllDesigners()
        gui.Utils.modified(atomic=True)        
    
    def deleteWaypoint(self, event):
        item = event.widget.find_closest(event.x+1, event.y+1)
        itemID = self.canvas.gettags(item)[1]
        itemNo = self.canvas.gettags(item)[2]
        waypoints = DataBoard.waypointSets[itemID].waypoints
        del waypoints[int(itemNo)-1]
        if len(waypoints) == 0:
            del DataBoard.waypointSets[itemID]
            DataBoard.selectedWaypointSet = ''
            DataBoard.selectedWaypoint = -1
        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
        
    def updateSelectedRegion(self, regionID = None):
        if not regionID == None:
            DataBoard.selectedRegion = regionID
        gui.Utils.updateRegionEditors()

    def updateSelectedWaypointSet(self):
        gui.Utils.updateWaypointEditors()
    
    def markSelectionInDesigners(self):
        self.canvas.itemconfigure("region", outline=self.regionOutline, width=self.unselectedOutlineWidth)
        self.canvas.itemconfigure(DataBoard.selectedRegion, outline=self.selectedRegionOutline, width=self.selectedOutlineWidth)
        self.canvas.itemconfigure("waypointCircle", outline=self.waypointOutline, width=self.unselectedOutlineWidth)
        self.canvas.itemconfigure("WC" + DataBoard.selectedWaypointSet + str(DataBoard.selectedWaypoint), outline=self.selectedWaypointOutline, width=self.selectedOutlineWidth)
    
    def removeRobotImage(self):
        self.canvas.delete('robotImage')
        self.drawingRobot = False
       
    def createStatusBar(self):
        status = gui.StatusBar.StatusBar(self)
        status.pack(side=BOTTOM, fill=X)
        self.statusBar = status
    
    
    def updateStatusBar(self, info=None):
        if not info == None:
            self.statusBar.set(info)
            return
        if self.editingObject.get() == 1: # editing region
            if DataBoard.selectedRegion == '':
                return
            reg = DataBoard.regions.getRegionByID(DataBoard.selectedRegion)
            if reg == None:
                self.statusBar.set('') # a boolean region has been selected; these are not visualized in graphical editors
                return
            self.statusBar.set(reg.name)
        else: # editing waypoint
            if DataBoard.selectedWaypointSet == '':
                return
            self.statusBar.set(DataBoard.waypointSets[DataBoard.selectedWaypointSet].name)

