'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on Dec 22, 2010

@author: Jimmy Li
'''

import Prototype
from Tkinter import *
import tkMessageBox
from config.Robots import Robots
from control.DataBoard import DataBoard
from control.Region import Region
from control.Waypoint import *
import gui.dialogs
import operator
import copy

class Designer1D(Prototype.DesignerTK):
    def __init__(self, master, settings): 
        self.dimensions = 1
        self.leftMarginWidth = 50 # Width of the space displaying the axis
        self.drawPadWidth = 50
        self.drawPadHeight = 600
        self.rightMarginWidth = 50
        self.canvasWidth = 150
        self.canvasHeight = 600
        self.drawingRegionIndex = 0
        
        Prototype.DesignerTK.__init__(self, master, settings)
        
        self.attributeFrame = Frame(self, background="#D9D9D9")
        self.attributeFrame.pack(side=TOP, fill=X)
        
        self.editingObjectRadioFrame = Frame(self.attributeFrame)
        self.editingObjectRadioFrame.grid(row=0, column=0, columnspan=2, padx=5, pady=0)
        
        self.editingObjectRegion = Radiobutton(self.editingObjectRadioFrame, text="Region", variable=self.editingObject, value=1)
        self.editingObjectRegion.pack(side=LEFT)
        self.editingObjectWaypoint = Radiobutton(self.editingObjectRadioFrame, text="Waypoint", variable=self.editingObject, value=2)
        self.editingObjectWaypoint.pack(side=LEFT)
        
        
        Label(self.attributeFrame, text="Edit").grid(sticky=E, row=1, column=0, padx=0, pady=2)
        self.editingParamLabels['X'] = StringVar(self.attributeFrame)        
        
        canvasframe = Frame(self, background="#F0F0F0")
        self.canvasframe = canvasframe
        self.canvas = Canvas(canvasframe, highlightthickness=0, width=self.canvasWidth, height=(self.canvasHeight-1), bg='white')
        self.canvas.create_rectangle(0,0,self.leftMarginWidth,self.canvasHeight, fill="#F0F0F0", outline = "#F0F0F0")
        self.canvas.create_rectangle(self.leftMarginWidth,0,self.leftMarginWidth+self.drawPadWidth,self.canvasHeight, fill="#FFFFFF", outline = "#FFFFFF", tags=("drawPad"))
                
        self.canvas.create_rectangle(self.leftMarginWidth+self.drawPadWidth,0,self.canvasWidth,self.canvasHeight, fill="#F0F0F0", outline = "#F0F0F0", tags=("rightMargin"))
        self.canvas.pack()
        canvasframe.pack(side=TOP, fill=BOTH)
        
        self.createStatusBar()
        self.registerListeners()        
        self.drawAxis()
        self.processSettings()
    
    def updateEditingParam(self):
        if self.editingParamCreated:
            self.editingParamMenu.grid_forget()
            
        self.editingParamLabels['X'].set(self.editingParameters['X'] + ' (' + Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars'][self.editingParameters['X']]['unit'] + ')') # default value
        self.editingParamMenu = OptionMenu(self.attributeFrame, self.editingParamLabels['X'], *tuple(self.visualizingParamLabels['X']), command=self.changeEditingParam)
        self.editingParamMenu.grid(row=1, sticky=W, column=1, pady=2, padx=0)
        self.editingParamCreated = True
            
    
    def changeEditingParam(self, event):
        ind = self.visualizingParamLabels['X'].index(self.editingParamLabels['X'].get())
        self.editingParameters['X'] = self.visualizingParameters['X'][ind]
    
    def settingsDialog(self):
        d = gui.dialogs.DesignerSettings.DesignerSettings1D(self)
        self.settings = d.result
        if self.settings == None:
            return
        self.processSettings()
        
    def initVisualization(self):
        self.visualizingRegions = {}
        for param in self.visualizingParameters['X']: 
            for reg in DataBoard.regions.getReleventRegions([param]):
                self.visualizingRegions[reg.id] = reg
        self.removeRobotImage()
                
    def redrawVisualization(self):
        self.canvas.delete("region")
        
        orderedBySize = []
        
        for regID in self.visualizingRegions:
            area = 0
            reg = self.visualizingRegions[regID]
            for constr in reg.constraints:
                if constr in self.visualizingParameters['X']:
                    upperbound = reg.constraints[constr][1]
                    if upperbound == 'inf':
                        top = -1
                    else:
                        top = self.unitToPixel(reg.constraints[constr][1])
                    
                    lowerbound = reg.constraints[constr][0]
                    if lowerbound == 'inf':
                        bottom = self.drawPadHeight+1
                    else:
                        bottom = self.unitToPixel(reg.constraints[constr][0])
                    self.canvas.create_rectangle(53, top, 97, bottom, fill=reg.color, outline=self.regionOutline, stipple="gray25", tags=("region", reg.id))
                    area += (bottom-top)
            orderedBySize.append((area, reg.id))
        
        orderedBySize = sorted(orderedBySize, key=operator.itemgetter(0), reverse=True)

        for entry in orderedBySize:
            self.canvas.tag_raise(entry[1])
                
        self.canvas.delete("waypoint")
        
        for setID in DataBoard.waypointSets:
            waypointSet = DataBoard.waypointSets[setID]
            for i in range(0, len(waypointSet.waypoints)):
                for param in self.visualizingParameters['X']:
                    if waypointSet.waypoints[i].position.has_key(param):
                        position = self.unitToPixel(waypointSet.waypoints[i].position[param])
                        self.canvas.create_oval(65, position-10, 85, position+10, fill=waypointSet.color, tags=("waypoint", waypointSet.id, str(i+1), "waypointCircle", "WC"+waypointSet.id+str(i+1)))
                        self.canvas.create_text(76, position-7, anchor=N, text=str(i+1), tags=("waypoint", waypointSet.id, str(i+1)))

        self.markSelectionInDesigners()
        self.updateStatusBar()
    
    def redrawRobotPosition(self, position):
        if not self.drawingRobot:
            self.canvas.delete('robotImage')
        
        for param in self.visualizingParameters['X']:
            index = self.visualizingParameters['X'].index(param)
            if position.has_key(param):
                val = position[param]
                pix = self.unitToPixel(val)
                if not self.drawingRobot:
                    #self.canvas.create_image(self.leftMarginWidth+25, pix, image=self.robotImage, tags=('robotImage','robotImage'+str(index)))
                    self.canvas.create_oval(self.leftMarginWidth + (self.drawPadWidth/2) - self.robotImageRadius, pix-self.robotImageRadius, self.leftMarginWidth + (self.drawPadWidth/2) + self.robotImageRadius, pix+self.robotImageRadius, fill = self.robotImageColor, outline=self.robotImageColor, tags=('robotImage','robotImage'+str(index)))
                else:
                    self.canvas.move('robotImage'+str(index), 0, pix - self.robotImagePos['X'][param])
                self.robotImagePos['X'][param] = pix
        
        self.canvas.tag_raise('robotImage')
            
        self.drawingRobot = True
        
    def drawCursor(self, event):
        if self.showCursor:
            y = self.canvas.canvasy(event.y)
            self.canvas.delete("cursor")
            self.canvas.create_line(self.leftMarginWidth+1, y, self.leftMarginWidth+self.drawPadWidth, y, fill="#FF3333", tags=("cursor"))
            measurement = (self.zeroY-y)/self.pixelPerUnit;
            self.canvas.create_text(self.leftMarginWidth+self.drawPadWidth+3, y, text="%.2f" % measurement, fill="#FF3333", anchor=W, tags=("cursor"))
    
    def setZoomIndex(self, event):
        Prototype.DesignerTK.setZoomIndex(self, event)
        self.centerPixelY = self.canvas.canvasx(event.y)
        self.centerUnitY = self.pixelToUnit(self.centerPixelY)

    def zoomRedraw(self):
        if self.pixelPerUnit != self.pixelPerUnitPrev:
            self.zeroY = self.centerUnitY*self.pixelPerUnit + self.centerPixelY # Derived from pixelToUnit()
        Prototype.DesignerTK.zoomRedraw(self)

    def initDrawRegion(self, event):
        self.drawingRegionIndex = self.canvas.canvasy(event.y)
        self.drawingRegion = True
        
    def drawRegion(self, event):
        self.drawCursor(event)
        canvasy = self.canvas.canvasy(event.y)
        length = canvasy - self.drawingRegionIndex
        self.canvas.delete("drawingRegion")
        if length > 0:
            self.canvas.create_rectangle(53, self.drawingRegionIndex, 97, canvasy, fill="#FFFF66", outline="#FFFF66", tags=("drawingRegion", "cursorActive"))
        elif length < 0:
            self.canvas.create_rectangle(53, canvasy, 97, self.drawingRegionIndex, fill="#FFFF66", outline="#FFFF66", tags=("drawingRegion", "cursorActive"))
    
    def endDrawRegion(self, event):
        
        if self.drawingRegion:
            self.drawingRegion = False
            self.canvas.delete("drawingRegion")
            canvasy = self.canvas.canvasy(event.y)
            
            if not canvasy == self.drawingRegionIndex:
                newRegion = Region()
                newRegion.setConstraint({self.editingParameters['X'] : [self.pixelToUnit(canvasy), self.pixelToUnit(self.drawingRegionIndex)]})
                DataBoard.regions.addRegion(newRegion)
                self.updateSelectedRegion(newRegion.id)
                gui.Utils.reinitAllDesigners()
                gui.Utils.modified(atomic=True)
                
    def endAddRegionDim(self, event):
        if self.drawingRegion:
            self.drawingRegion = False
            self.canvas.delete("drawingRegion")  
            
            # Do not add dimensions to the selected region if the region already has constraints
            # in the current designer
            region = DataBoard.regions.getRegionByID(DataBoard.selectedRegion)
            if region == None:
                return
            
            canvasy = self.canvas.canvasy(event.y)
            
            if not canvasy == self.drawingRegionIndex:        
                DataBoard.regions.removeRegion(region)                
                region.setConstraint({self.editingParameters['X'] : [self.pixelToUnit(canvasy), self.pixelToUnit(self.drawingRegionIndex)]})
                DataBoard.regions.addRegion(region)
                self.updateSelectedRegion(region.id)
                gui.Utils.reinitAllDesigners()
                gui.Utils.modified(atomic=True)   
    
    def deleteRegionDim(self, event):
        
        item = event.widget.find_closest(event.x+1, event.y+1)
        itemID = self.canvas.gettags(item)[1]        
        reg = DataBoard.regions.getRegionByID(itemID)
        DataBoard.regions.removeRegion(reg)
        reg.setConstraint({self.editingParameters['X']: None})
        DataBoard.regions.addRegion(reg)
        gui.Utils.updateRegionEditors()
        gui.Utils.reinitAllDesigners()
        gui.Utils.modified(atomic=True)
    
    def deleteWaypointDim(self, event):
        
        item = event.widget.find_closest(event.x+1, event.y+1)
        itemID = self.canvas.gettags(item)[1]
        itemNo = self.canvas.gettags(item)[2]
        waypoints = DataBoard.waypointSets[itemID].waypoints
        waypoint = waypoints[int(itemNo)-1]
        del waypoint.position[self.editingParameters['X']]
        self.updateSelectedWaypointSet()
        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
    
    def moveWaypoint(self, event):
        self.drawCursor(event)
        canvasy = self.canvas.canvasy(event.y)
        self.canvas.delete("movingWaypoint")
        self.canvas.create_oval(65, canvasy-10, 85, canvasy+10, fill="#FFFF66", outline="#FFFF66", tags=("movingWaypoint"))
    
    def endMovingWaypoint(self, event):
        self.movingWaypoint = False
        canvasy = self.canvas.canvasy(event.y)
        if self.moveWaypointIndexY == canvasy:
            return
        self.canvas.delete("movingWaypoint")
        if not self.checkEditingParamWritable():
            return
        
        waypointSetID = self.canvas.gettags(self.waypointBeingMoved)[1]
        waypointNo = self.canvas.gettags(self.waypointBeingMoved)[2]
        waypoint = DataBoard.waypointSets[waypointSetID].waypoints[int(waypointNo)-1]
        waypoint.position[self.editingParameters['X']] = self.pixelToUnit(canvasy)

        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)

    
    def createNewWaypointSet(self, event):
        if not self.checkEditingParamWritable():
            return
        
        canvasy = self.canvas.canvasy(event.y)
        newWaypointSet = WaypointSet()
        waypoint = Waypoint()
        waypoint.position[self.editingParameters['X']] = self.pixelToUnit(canvasy)
        newWaypointSet.waypoints.append(waypoint)
        DataBoard.waypointSets[newWaypointSet.id] = newWaypointSet
        DataBoard.selectedWaypointSet = newWaypointSet.id
        DataBoard.selectedWaypoint = 1
        self.updateSelectedWaypointSet()
        gui.Utils.redrawAllDesigners()      
        gui.Utils.modified(atomic=True)  
    
    def addToWaypointSet(self, event):
        if not self.checkEditingParamWritable():
            return
        canvasy = self.canvas.canvasy(event.y)
        if not DataBoard.waypointSets.has_key(DataBoard.selectedWaypointSet):
            return
        
        waypointSet = DataBoard.waypointSets[DataBoard.selectedWaypointSet]
        waypoint = Waypoint()
        waypoint.position[self.editingParameters['X']] = self.pixelToUnit(canvasy)
        index = int(DataBoard.selectedWaypoint)
        waypointSet.waypoints.insert(index, waypoint)
        DataBoard.selectedWaypoint = index + 1
        self.updateSelectedWaypointSet()
        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
    
    def addToWaypointDim(self, event):
        if not self.checkEditingParamWritable():
            return
        canvasy = self.canvas.canvasy(event.y)
        if not DataBoard.waypointSets.has_key(DataBoard.selectedWaypointSet):
            return
        
        waypointSet = DataBoard.waypointSets[DataBoard.selectedWaypointSet]
        waypointIndex = int(DataBoard.selectedWaypoint) - 1
        waypoint = waypointSet.waypoints[waypointIndex]
        waypoint.position[self.editingParameters['X']] = self.pixelToUnit(canvasy)
        gui.Utils.updateWaypointEditors()
        gui.Utils.redrawAllDesigners()
        gui.Utils.modified(atomic=True)
    
    
    def updateIfVisualizing(self, parameters):
        reInited = False
        for param in self.visualizingParameters['X']:
            if param in parameters:
                self.initVisualization()
                self.redrawVisualization()
                reInited = True
                break
        if not reInited:
            self.redrawVisualization()
    
    def pixelToUnit(self, pixel):
        return (self.zeroY - pixel) / self.pixelPerUnit
    
    def unitToPixel(self, unit):
        return self.zeroY - (unit * self.pixelPerUnit)
    
    def drawAxis(self):
        self.drawVerticalAxis(self.leftMarginWidth)
        
    def checkEditingParamWritable(self):
        stateVarsDic = Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars']
        if not stateVarsDic[self.editingParameters['X']]['writable']:
            tkMessageBox.showerror("Error", "You are editing " + self.editingParameters['X'] + ", which is not a writable state variable.")
            return False
        return True
