'''
Copyright (c) 2011, Mobile Robotics Lab, McGill University
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the Mobile Robotics Lab, McGill University nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL MOBILE ROBOTICS LAB, MCGILL UNIVERSITY BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

'''
Created on 2010-07-28

@author: Jimmy Li
'''

import threading
import time
from control.DataBoard import DataBoard
import math
import control.Runtime
from config.Robots import Robots

class TaskThread(threading.Thread):
    '''
    classdocs
    '''

    def __init__(self, region, wayptSet):
        '''
        Constructor
        '''
        threading.Thread.__init__(self)
        
        self.lock = threading.RLock()
        
        self.region = region
        self.wayptSet = wayptSet
        
        self.onWaypoint = 0
        
        #requested state changes
        self.reqStateChange = {}
        
        self.priority = region.priority
                
        self.runTimeEnv = control.Runtime.Environment(region, self, 'Waypt Set ' + self.wayptSet.name)
        
        self.description = wayptSet.name + ' wpt controller thread'
        
        self.defaultError = 2
        
        self.alive = True
        
        self.kill = False
    
    def setDestination(self):
        self.lock.acquire()
        for param in self.wayptSet.waypoints[self.onWaypoint].position:
            self.reqStateChange[param] = self.wayptSet.waypoints[self.onWaypoint].position[param]
        self.lock.release()
    
    def run(self):
        # TODO: This function needs some cleaning up after we do some testing
        # (delete redundant code and testing statements)
        
        self.lock.acquire()
        control.DataBoard.DataBoard.metaControllerHandle.log('WAYPT', 'Activating '+self.wayptSet.name)
        self.lock.release()
        
        while (1):
            if not control.StateBoard.StateBoard.metaControlStarted:
                break
            
            if self.kill:
                break
            
            if not self.region.inside and self.region.bounded:
                time.sleep(0.1)
                continue
           
            currPosition = {}
            gotPosition = True
            
            self.lock.acquire()
            for param in self.wayptSet.waypoints[self.onWaypoint].position:
                if not control.StateBoard.StateBoard.currentState.has_key(param):
                    gotPosition = False
                    break
                else:    
                    currPosition[param] = control.StateBoard.StateBoard.currentState[param]
            
            self.lock.release()
            if not gotPosition:
                time.sleep(0.1)
                continue
            
            #See if we've gotten to a waypoint
            atWaypoint = True
            for param in self.wayptSet.waypoints[self.onWaypoint].position:
                current = currPosition[param]
                destination = self.wayptSet.waypoints[self.onWaypoint].position[param]
                try:
                    error = Robots.stateVarSets[DataBoard.currentStateVarSet]['stateVars'][param]['error']
                except KeyError:
                    error = self.defaultError
                    self.lock.acquire()
                    DataBoard.metaControllerHandle.log('WARN', 'State-variable ' + param + ' has no error attribute. Using ' + error + ' as default.')
                    self.lock.release()
                
                if math.fabs(destination - current) > error:
                    atWaypoint = False
                    break
                
            if not atWaypoint:
                self.setDestination()
            else: 
                self.lock.acquire()
                DataBoard.metaControllerHandle.log('WAYPT', 'Arrived at ' + self.wayptSet.name + ' waypoint '+ str(self.onWaypoint+1))
                DataBoard.metaControllerHandle.log('CODE', 'Executing code block of waypt set ' + self.wayptSet.name + ' waypoint '+ str(self.onWaypoint+1))
                self.lock.release()
                
                self.runTimeEnv.runCodeBlock(self.wayptSet.waypoints[self.onWaypoint].commands+"\n")
                
                self.onWaypoint += 1
                if self.onWaypoint >= len(self.wayptSet.waypoints):
                    break
                self.setDestination()
            time.sleep(0.1)
        
        self.lock.acquire()
        DataBoard.metaControllerHandle.log('WAYPT', 'Deactivating ' + self.wayptSet.name)
        self.lock.release()
                
        self.lock.acquire()
        self.reqStateChange.clear()
        self.lock.release()
        
        self.alive = False
        
